<?php

namespace App\Console\Commands;

use App\Enums\ExamAttemptStatus;
use App\Models\ExamAnswer;
use App\Models\ExamAttempt;
use App\Models\ExamSession;
use App\Models\QuestionBank;
use App\Models\Student;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class LoadTestExam extends Command
{
    protected $signature = 'load-test:exam 
                            {--users=50 : Number of concurrent users to simulate}
                            {--session= : Exam session ID to test (uses first active if not specified)}
                            {--create-students : Create test students if needed}
                            {--cleanup : Clean up test data after running}';

    protected $description = 'Load test the exam system by simulating concurrent users taking exams';

    private array $metrics = [
        'total_requests' => 0,
        'successful_requests' => 0,
        'failed_requests' => 0,
        'response_times' => [],
        'errors' => [],
    ];

    public function handle(): int
    {
        $userCount = (int) $this->option('users');
        $sessionId = $this->option('session');
        $createStudents = $this->option('create-students');
        $cleanup = $this->option('cleanup');

        $this->info("🚀 CBT System Load Test");
        $this->info("========================");
        $this->newLine();

        // Get or find exam session
        $session = $sessionId 
            ? ExamSession::find($sessionId)
            : ExamSession::where('status', 'active')->first();

        if (!$session) {
            $this->error('No active exam session found. Please create one or specify --session=ID');
            return Command::FAILURE;
        }

        $this->info("📝 Exam Session: {$session->title} (ID: {$session->id})");
        $this->info("📚 Exam: {$session->exam->title}");
        $this->info("❓ Questions: {$session->exam->total_questions}");
        $this->info("👥 Simulating: {$userCount} concurrent users");
        $this->newLine();

        // Get or create students
        $students = $this->getOrCreateStudents($userCount, $createStudents);
        
        if ($students->count() < $userCount) {
            $this->warn("Only {$students->count()} students available. Use --create-students to create more.");
            $userCount = $students->count();
        }

        if ($userCount === 0) {
            $this->error('No students available for testing.');
            return Command::FAILURE;
        }

        // Clean up any existing attempts for these students in this session
        $this->info("🧹 Cleaning up previous test attempts...");
        ExamAttempt::where('exam_session_id', $session->id)
            ->whereIn('candidate_id', $students->pluck('id'))
            ->where('candidate_type', Student::class)
            ->delete();

        $this->newLine();
        $this->info("🏃 Starting load test...");
        $this->newLine();

        $progressBar = $this->output->createProgressBar($userCount);
        $progressBar->start();

        $startTime = microtime(true);

        // Simulate concurrent users using parallel processes
        $results = $this->runConcurrentTests($students->take($userCount), $session, $progressBar);

        $progressBar->finish();
        $this->newLine(2);

        $totalTime = microtime(true) - $startTime;

        // Display results
        $this->displayResults($totalTime, $userCount);

        // Cleanup if requested
        if ($cleanup) {
            $this->info("🧹 Cleaning up test data...");
            ExamAttempt::where('exam_session_id', $session->id)
                ->whereIn('candidate_id', $students->pluck('id'))
                ->where('candidate_type', Student::class)
                ->delete();
            
            if ($createStudents) {
                Student::where('matriculation_number', 'like', 'LOADTEST/%')->delete();
            }
            $this->info("✅ Cleanup complete.");
        }

        return Command::SUCCESS;
    }

    private function getOrCreateStudents(int $count, bool $createIfNeeded)
    {
        $students = Student::where('is_active', true)->take($count)->get();

        if ($students->count() < $count && $createIfNeeded) {
            $toCreate = $count - $students->count();
            $this->info("Creating {$toCreate} test students...");

            // Find the highest existing load test student number
            $lastLoadTestStudent = Student::where('matriculation_number', 'like', 'LOADTEST/%')
                ->orderByRaw("CAST(SUBSTRING(matriculation_number FROM 'LOADTEST/(.*)') AS INTEGER) DESC")
                ->first();
            
            $startNum = 1;
            if ($lastLoadTestStudent) {
                preg_match('/LOADTEST\/(\d+)/', $lastLoadTestStudent->matriculation_number, $matches);
                $startNum = (int)($matches[1] ?? 0) + 1;
            }

            for ($i = 0; $i < $toCreate; $i++) {
                $num = $startNum + $i;
                $student = Student::create([
                    'full_name' => "Load Test Student {$num}",
                    'matriculation_number' => 'LOADTEST/' . str_pad($num, 4, '0', STR_PAD_LEFT),
                    'email' => "loadtest{$num}@test.com",
                    'password' => bcrypt('password'),
                    'session' => '2024/2025',
                    'is_active' => true,
                ]);
                $students->push($student);
            }
        }

        return $students;
    }

    private function runConcurrentTests($students, ExamSession $session, $progressBar): array
    {
        $results = [];

        foreach ($students as $student) {
            $result = $this->simulateExamAttempt($student, $session);
            $results[] = $result;
            $progressBar->advance();
        }

        return $results;
    }

    private function simulateExamAttempt(Student $student, ExamSession $session): array
    {
        $result = [
            'student_id' => $student->id,
            'success' => false,
            'operations' => [],
        ];

        try {
            // 1. Start exam attempt
            $startTime = microtime(true);
            
            $exam = $session->exam;
            $questionIds = $exam->getQuestionsForAttempt();
            $questions = QuestionBank::whereIn('id', $questionIds)->get();
            
            $attempt = ExamAttempt::create([
                'exam_session_id' => $session->id,
                'candidate_type' => Student::class,
                'candidate_id' => $student->id,
                'question_order' => $questionIds,
                'started_at' => now(),
                'total_questions' => count($questionIds),
                'status' => ExamAttemptStatus::IN_PROGRESS,
                'current_question_index' => 0,
                'ip_address' => '127.0.0.1',
                'user_agent' => 'LoadTest/1.0',
            ]);

            $this->recordOperation('start_exam', microtime(true) - $startTime);

            // 2. Answer questions (simulate answering each question)
            foreach ($questions as $index => $question) {
                $answerStart = microtime(true);
                
                // Get a random option as the answer
                $options = $question->options ?? [];
                $optionKeys = array_keys($options);
                $selectedAnswer = !empty($optionKeys) ? $optionKeys[array_rand($optionKeys)] : 'A';
                $isCorrect = $selectedAnswer === $question->correct_answer;
                $marksPossible = (int) ($exam->marks_per_question ?? 1);

                ExamAnswer::updateOrCreate(
                    [
                        'exam_attempt_id' => $attempt->id,
                        'question_bank_id' => $question->id,
                    ],
                    [
                        'question_index' => $index,
                        'selected_answer' => $selectedAnswer,
                        'correct_answer' => $question->correct_answer,
                        'is_correct' => $isCorrect,
                        'marks_obtained' => $isCorrect ? $marksPossible : 0,
                        'marks_possible' => $marksPossible,
                        'time_spent_seconds' => rand(5, 30),
                        'answered_at' => now(),
                        'is_flagged' => false,
                        'is_skipped' => false,
                    ]
                );

                $attempt->update([
                    'current_question_index' => $index + 1,
                    'answered_questions' => $index + 1,
                ]);

                $this->recordOperation('answer_question', microtime(true) - $answerStart);
            }

            // 3. Submit exam
            $submitStart = microtime(true);
            
            $correctAnswers = ExamAnswer::where('exam_attempt_id', $attempt->id)
                ->where('is_correct', true)
                ->count();
            
            $totalQuestions = $questions->count();
            $wrongAnswers = $totalQuestions - $correctAnswers;
            $scoreObtained = $correctAnswers * ($exam->marks_per_question ?? 1);
            $maxScore = $totalQuestions * ($exam->marks_per_question ?? 1);
            $scorePercentage = $maxScore > 0 ? round(($scoreObtained / $maxScore) * 100, 2) : 0;

            $attempt->update([
                'completed_at' => now(),
                'status' => ExamAttemptStatus::COMPLETED,
                'correct_answers' => $correctAnswers,
                'wrong_answers' => $wrongAnswers,
                'skipped_questions' => 0,
                'score_obtained' => $scoreObtained,
                'score_percentage' => $scorePercentage,
                'passed' => $scorePercentage >= ($exam->passing_score ?? 50),
                'time_spent_seconds' => rand(60, 300),
            ]);

            $this->recordOperation('submit_exam', microtime(true) - $submitStart);

            $result['success'] = true;
            $this->metrics['successful_requests']++;

        } catch (\Exception $e) {
            $result['error'] = $e->getMessage();
            $this->metrics['failed_requests']++;
            $this->metrics['errors'][] = $e->getMessage();
            Log::error('Load test error', ['student' => $student->id, 'error' => $e->getMessage()]);
        }

        $this->metrics['total_requests']++;
        return $result;
    }

    private function recordOperation(string $operation, float $duration): void
    {
        if (!isset($this->metrics['response_times'][$operation])) {
            $this->metrics['response_times'][$operation] = [];
        }
        $this->metrics['response_times'][$operation][] = $duration * 1000; // Convert to ms
    }

    private function displayResults(float $totalTime, int $userCount): void
    {
        $this->info("📊 LOAD TEST RESULTS");
        $this->info("====================");
        $this->newLine();

        // Summary
        $this->table(
            ['Metric', 'Value'],
            [
                ['Total Users Simulated', $userCount],
                ['Successful Completions', $this->metrics['successful_requests']],
                ['Failed Completions', $this->metrics['failed_requests']],
                ['Success Rate', round(($this->metrics['successful_requests'] / max($this->metrics['total_requests'], 1)) * 100, 2) . '%'],
                ['Total Time', round($totalTime, 2) . ' seconds'],
                ['Throughput', round($userCount / $totalTime, 2) . ' users/second'],
            ]
        );

        $this->newLine();
        $this->info("⏱️  Response Times by Operation");
        $this->info("================================");

        $responseData = [];
        foreach ($this->metrics['response_times'] as $operation => $times) {
            if (empty($times)) continue;
            
            sort($times);
            $count = count($times);
            $avg = array_sum($times) / $count;
            $min = min($times);
            $max = max($times);
            $p95 = $times[(int)($count * 0.95)] ?? $max;
            $p99 = $times[(int)($count * 0.99)] ?? $max;

            $responseData[] = [
                ucfirst(str_replace('_', ' ', $operation)),
                round($avg, 2) . ' ms',
                round($min, 2) . ' ms',
                round($max, 2) . ' ms',
                round($p95, 2) . ' ms',
                round($p99, 2) . ' ms',
            ];
        }

        $this->table(
            ['Operation', 'Avg', 'Min', 'Max', 'P95', 'P99'],
            $responseData
        );

        // Errors if any
        if (!empty($this->metrics['errors'])) {
            $this->newLine();
            $this->error("❌ Errors encountered:");
            foreach (array_unique($this->metrics['errors']) as $error) {
                $this->line("  - " . $error);
            }
        }

        $this->newLine();
        
        // Performance assessment
        $avgResponseTime = 0;
        $totalTimes = [];
        foreach ($this->metrics['response_times'] as $times) {
            $totalTimes = array_merge($totalTimes, $times);
        }
        if (!empty($totalTimes)) {
            $avgResponseTime = array_sum($totalTimes) / count($totalTimes);
        }

        $this->info("📈 Performance Assessment");
        $this->info("=========================");
        
        if ($this->metrics['failed_requests'] === 0 && $avgResponseTime < 100) {
            $this->info("✅ EXCELLENT - System handled {$userCount} concurrent users with no failures and fast response times.");
        } elseif ($this->metrics['failed_requests'] === 0 && $avgResponseTime < 500) {
            $this->info("✅ GOOD - System handled {$userCount} concurrent users with no failures.");
        } elseif ($this->metrics['failed_requests'] < $userCount * 0.05) {
            $this->warn("⚠️  ACCEPTABLE - System handled most users but had some failures (" . round(($this->metrics['failed_requests'] / $userCount) * 100, 1) . "% failure rate).");
        } else {
            $this->error("❌ NEEDS IMPROVEMENT - High failure rate detected. Consider optimizing database queries or increasing server resources.");
        }

        $this->newLine();
        $this->info("💡 Recommendations:");
        
        if ($avgResponseTime > 500) {
            $this->line("  - Consider adding database indexes");
            $this->line("  - Enable query caching");
            $this->line("  - Use Redis for session storage");
        }
        
        if ($this->metrics['failed_requests'] > 0) {
            $this->line("  - Check database connection pool settings");
            $this->line("  - Review error logs for specific issues");
        }

        $this->line("  - For production: test with --users=100, --users=200, --users=500");
        $this->line("  - Monitor server CPU and memory during tests");
    }
}
