<?php

namespace App\Filament\Pages;

use App\Enums\NavigationGroup;
use App\Models\ExamAnswer;
use App\Models\ExamAttempt;
use BackedEnum;
use Filament\Pages\Page;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;
use UnitEnum;

class EssayGrading extends Page implements Forms\Contracts\HasForms
{
    use Forms\Concerns\InteractsWithForms;

    protected static string|BackedEnum|null $navigationIcon = 'heroicon-o-pencil-square';
    protected static ?string $navigationLabel = 'Grade Essays';
    protected static ?string $title = 'Essay Grading';
    protected static ?string $slug = 'essay-grading';
    protected static string|UnitEnum|null $navigationGroup = NavigationGroup::CBT_MANAGEMENT->value;
    protected static ?int $navigationSort = 5;
    protected static bool $shouldRegisterNavigation = false;

    protected string $view = 'filament.pages.essay-grading';

    public ?ExamAnswer $currentAnswer = null;
    public ?array $formData = [];
    public int $pendingCount = 0;
    public int $gradedCount = 0;
    public array $pendingAnswers = [];
    public ?int $selectedAnswerId = null;

    public static function canAccess(): bool
    {
        // Allow any authenticated admin user to access essay grading
        return Auth::check();
    }

    public function mount(): void
    {
        $this->loadCounts();
        $this->loadPendingAnswers();
    }

    protected function loadCounts(): void
    {
        $query = ExamAnswer::query()
            ->where('requires_grading', true);

        $this->pendingCount = (clone $query)->where('is_graded', false)->count();
        $this->gradedCount = (clone $query)->where('is_graded', true)->count();
    }

    protected function loadPendingAnswers(): void
    {
        $query = ExamAnswer::query()
            ->with(['question', 'attempt.examSession.exam', 'attempt.candidate'])
            ->where('requires_grading', true)
            ->where('is_graded', false);

        $this->pendingAnswers = $query
            ->orderBy('created_at', 'asc')
            ->limit(50)
            ->get()
            ->map(function ($answer) {
                $candidate = $answer->attempt->candidate;
                return [
                    'id' => $answer->id,
                    'exam_title' => $answer->attempt->exam->title ?? 'Unknown Exam',
                    'question_preview' => \Illuminate\Support\Str::limit($answer->question->question_text ?? '', 50),
                    'candidate_name' => $candidate->full_name ?? $candidate->name ?? 'Unknown',
                    'submitted_at' => $answer->answered_at?->diffForHumans() ?? 'N/A',
                    'marks_possible' => $answer->marks_possible,
                ];
            })
            ->toArray();
    }

    public function selectAnswer(int $answerId): void
    {
        $this->selectedAnswerId = $answerId;
        $this->currentAnswer = ExamAnswer::with(['question', 'attempt.examSession.exam', 'attempt.candidate', 'grader'])
            ->find($answerId);

        if ($this->currentAnswer) {
            $this->formData = [
                'marks_obtained' => $this->currentAnswer->marks_obtained ?? 0,
                'grading_feedback' => $this->currentAnswer->grading_feedback ?? '',
            ];
        }
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('marks_obtained')
                    ->label('Marks Awarded')
                    ->numeric()
                    ->required()
                    ->minValue(0)
                    ->maxValue($this->currentAnswer?->marks_possible ?? 100)
                    ->suffix('/ ' . ($this->currentAnswer?->marks_possible ?? 0))
                    ->helperText('Maximum: ' . ($this->currentAnswer?->marks_possible ?? 0) . ' marks'),

                Forms\Components\Textarea::make('grading_feedback')
                    ->label('Feedback to Candidate')
                    ->rows(4)
                    ->placeholder('Provide constructive feedback on the answer...')
                    ->helperText('This feedback will be shown to the candidate when they view their results.'),
            ])
            ->statePath('formData');
    }

    public function submitGrade(): void
    {
        if (!$this->currentAnswer) {
            Notification::make()
                ->title('Error')
                ->body('No answer selected for grading.')
                ->danger()
                ->send();
            return;
        }

        $data = $this->form->getState();
        $user = Auth::user();

        // Update the answer with grade
        $this->currentAnswer->update([
            'marks_obtained' => $data['marks_obtained'],
            'grading_feedback' => $data['grading_feedback'],
            'is_graded' => true,
            'graded_by' => $user->id,
            'graded_at' => now(),
            'is_correct' => $data['marks_obtained'] > 0, // Consider any marks as partially correct
        ]);

        // Update the exam attempt stats
        $this->currentAnswer->attempt->updateStats();

        Notification::make()
            ->title('Grade Submitted')
            ->body('The essay answer has been graded successfully.')
            ->success()
            ->send();

        // Refresh the list
        $this->loadCounts();
        $this->loadPendingAnswers();
        $this->currentAnswer = null;
        $this->selectedAnswerId = null;
        $this->formData = [];
    }

    public function skipAnswer(): void
    {
        // Just deselect the current answer
        $this->currentAnswer = null;
        $this->selectedAnswerId = null;
        $this->formData = [];
    }
}
