<?php

namespace App\Filament\Pages;

use App\Settings\FeatureSettings;
use Filament\Actions\Action;
use Filament\Forms\Components\Toggle;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use BackedEnum;
use UnitEnum;

class ManageFeatureSettings extends Page
{
    protected static string|BackedEnum|null $navigationIcon = 'heroicon-o-cog-6-tooth';

    protected string $view = 'filament.pages.manage-settings';

    protected static string|UnitEnum|null $navigationGroup = 'Settings';

    protected static ?int $navigationSort = 14;

    protected static ?string $title = 'Feature Settings';

    protected static ?string $navigationLabel = 'Features';

    public ?array $data = [];

    public function mount(): void
    {
        $settings = app(FeatureSettings::class);
        
        $this->form->fill([
            'secondary_user_exams' => $settings->secondary_user_exams,
            'practice_mode' => $settings->practice_mode,
            'certificate_generation' => $settings->certificate_generation,
            'result_printing' => $settings->result_printing,
            'bulk_import' => $settings->bulk_import,
            'api_access' => $settings->api_access,
            'dark_mode' => $settings->dark_mode,
        ]);
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('Core Features')
                    ->description('Enable or disable major system features.')
                    ->schema([
                        Toggle::make('secondary_user_exams')
                            ->label('Secondary User Exams')
                            ->helperText('Allow secondary users to take exams'),

                        Toggle::make('practice_mode')
                            ->label('Practice Mode')
                            ->helperText('Enable practice/mock exam mode'),

                        Toggle::make('certificate_generation')
                            ->label('Certificate Generation')
                            ->helperText('Generate certificates for passed exams'),

                        Toggle::make('result_printing')
                            ->label('Result Printing')
                            ->helperText('Allow printing of exam results'),
                    ])
                    ->columns(2),

                Section::make('Advanced Features')
                    ->description('Additional system capabilities.')
                    ->schema([
                        Toggle::make('bulk_import')
                            ->label('Bulk Import')
                            ->helperText('Allow bulk importing of questions and users'),

                        Toggle::make('api_access')
                            ->label('API Access')
                            ->helperText('Enable API access for external integrations'),

                        Toggle::make('dark_mode')
                            ->label('Dark Mode')
                            ->helperText('Allow dark mode theme'),
                    ])
                    ->columns(2),
            ])
            ->statePath('data');
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('save')
                ->label('Save Settings')
                ->submit('save'),
        ];
    }

    public function save(): void
    {
        $data = $this->form->getState();
        
        $settings = app(FeatureSettings::class);
        $settings->secondary_user_exams = (bool) $data['secondary_user_exams'];
        $settings->practice_mode = (bool) $data['practice_mode'];
        $settings->certificate_generation = (bool) $data['certificate_generation'];
        $settings->result_printing = (bool) $data['result_printing'];
        $settings->bulk_import = (bool) $data['bulk_import'];
        $settings->api_access = (bool) $data['api_access'];
        $settings->dark_mode = (bool) $data['dark_mode'];
        $settings->save();

        Notification::make()
            ->title('Settings saved successfully')
            ->success()
            ->send();
    }
}
