<?php

namespace App\Filament\Pages;

use App\Settings\TerminologySettings;
use Filament\Actions\Action;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use BackedEnum;
use UnitEnum;

class ManageTerminologySettings extends Page
{
    protected static string|BackedEnum|null $navigationIcon = 'heroicon-o-language';

    protected string $view = 'filament.pages.manage-settings';

    protected static string|UnitEnum|null $navigationGroup = 'Settings';

    protected static ?int $navigationSort = 13;

    protected static ?string $title = 'Terminology Settings';

    protected static ?string $navigationLabel = 'Terminology';

    public ?array $data = [];

    public function mount(): void
    {
        $settings = app(TerminologySettings::class);
        
        $this->form->fill([
            'primary_user_singular' => $settings->primary_user_singular,
            'primary_user_plural' => $settings->primary_user_plural,
            'primary_user_id_label' => $settings->primary_user_id_label,
            'primary_user_id_field' => $settings->primary_user_id_field,
            'secondary_user_singular' => $settings->secondary_user_singular,
            'secondary_user_plural' => $settings->secondary_user_plural,
            'secondary_user_id_label' => $settings->secondary_user_id_label,
            'secondary_user_id_field' => $settings->secondary_user_id_field,
            'period_label' => $settings->period_label,
            'period_format' => $settings->period_format,
            'division_label' => $settings->division_label,
        ]);
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('Primary User Labels')
                    ->description('Customize how primary users (exam takers) are referred to.')
                    ->schema([
                        TextInput::make('primary_user_singular')
                            ->label('Singular')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., Student, Candidate, Employee'),

                        TextInput::make('primary_user_plural')
                            ->label('Plural')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., Students, Candidates, Employees'),

                        TextInput::make('primary_user_id_label')
                            ->label('ID Label')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., Student ID, Employee Number'),

                        TextInput::make('primary_user_id_field')
                            ->label('ID Field Name')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., student_id, employee_number'),
                    ])
                    ->columns(2),

                Section::make('Secondary User Labels')
                    ->description('Customize how secondary users (staff/examiners) are referred to.')
                    ->schema([
                        TextInput::make('secondary_user_singular')
                            ->label('Singular')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., Staff, Instructor, Examiner'),

                        TextInput::make('secondary_user_plural')
                            ->label('Plural')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., Staff, Instructors, Examiners'),

                        TextInput::make('secondary_user_id_label')
                            ->label('ID Label')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., Staff ID, Personnel Number'),

                        TextInput::make('secondary_user_id_field')
                            ->label('ID Field Name')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., staff_id, personnel_number'),
                    ])
                    ->columns(2),

                Section::make('Organizational Terms')
                    ->description('Customize organizational terminology.')
                    ->schema([
                        TextInput::make('period_label')
                            ->label('Period Label')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., Session, Semester, Term'),

                        TextInput::make('period_format')
                            ->label('Period Format')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., 2024/2025, Fall 2024'),

                        TextInput::make('division_label')
                            ->label('Division Label')
                            ->required()
                            ->maxLength(50)
                            ->helperText('e.g., Department, Unit, Division'),
                    ])
                    ->columns(3),
            ])
            ->statePath('data');
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('save')
                ->label('Save Settings')
                ->submit('save'),
        ];
    }

    public function save(): void
    {
        $data = $this->form->getState();
        
        $settings = app(TerminologySettings::class);
        $settings->primary_user_singular = $data['primary_user_singular'];
        $settings->primary_user_plural = $data['primary_user_plural'];
        $settings->primary_user_id_label = $data['primary_user_id_label'];
        $settings->primary_user_id_field = $data['primary_user_id_field'];
        $settings->secondary_user_singular = $data['secondary_user_singular'];
        $settings->secondary_user_plural = $data['secondary_user_plural'];
        $settings->secondary_user_id_label = $data['secondary_user_id_label'];
        $settings->secondary_user_id_field = $data['secondary_user_id_field'];
        $settings->period_label = $data['period_label'];
        $settings->period_format = $data['period_format'];
        $settings->division_label = $data['division_label'];
        $settings->save();

        Notification::make()
            ->title('Settings saved successfully')
            ->success()
            ->send();
    }
}
