<?php

namespace App\Filament\Resources\Applicants;

use App\Enums\NavigationGroup;
use App\Models\Applicant;
use App\Settings\TerminologySettings;
use BackedEnum;
use Filament\Actions\DeleteAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use UnitEnum;

class ApplicantResource extends Resource
{
    protected static ?string $model = Applicant::class;

    protected static string|BackedEnum|null $navigationIcon = 'heroicon-o-document-text';

    protected static ?string $recordTitleAttribute = 'full_name';

    protected static string|UnitEnum|null $navigationGroup = NavigationGroup::CBT_MANAGEMENT->value;

    protected static ?int $navigationSort = 3;

    public static function form(Schema $schema): Schema
    {
        $terminology = app(TerminologySettings::class);

        return $schema
            ->columns(1)
            ->components([
                Section::make('Applicant Information')
                    ->description('Enter the applicant details for recruitment assessment')
                    ->columns(2)
                    ->columnSpanFull()
                    ->schema([
                        TextInput::make('application_number')
                            ->label('Application Number')
                            ->required()
                            ->unique(ignoreRecord: true)
                            ->maxLength(50)
                            ->placeholder('e.g., APP-2026-001')
                            ->helperText('Unique identifier for the applicant'),

                        TextInput::make('full_name')
                            ->label('Full Name')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('e.g., John Doe'),

                        TextInput::make('email')
                            ->label('Email Address')
                            ->email()
                            ->unique(Applicant::class, 'email', ignoreRecord: true)
                            ->maxLength(255)
                            ->placeholder('e.g., applicant@email.com'),

                        TextInput::make('phone')
                            ->label('Phone Number')
                            ->tel()
                            ->maxLength(20)
                            ->placeholder('e.g., +234 801 234 5678'),

                        TextInput::make('position_applied')
                            ->label('Position Applied For')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('e.g., Medical Officer, Nurse, IT Support'),

                        TextInput::make('department')
                            ->label('Department')
                            ->maxLength(255)
                            ->placeholder('e.g., Psychiatry, Nursing, Administration'),

                        TextInput::make('password')
                            ->label('Password')
                            ->password()
                            ->required(fn (string $context): bool => $context === 'create')
                            ->minLength(6)
                            ->dehydrateStateUsing(fn (?string $state): ?string =>
                                filled($state) ? bcrypt($state) : null
                            )
                            ->dehydrated(fn (?string $state): bool => filled($state))
                            ->placeholder('Minimum 6 characters')
                            ->helperText('Password for applicant to login and take assessments'),

                        Toggle::make('is_active')
                            ->label('Active')
                            ->default(true)
                            ->helperText('Inactive applicants cannot login or take assessments'),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('application_number')
                    ->label('App No.')
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->badge()
                    ->color('info'),

                TextColumn::make('full_name')
                    ->label('Name')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('position_applied')
                    ->label('Position')
                    ->searchable()
                    ->sortable()
                    ->limit(30),

                TextColumn::make('department')
                    ->label('Department')
                    ->searchable()
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('email')
                    ->label('Email')
                    ->searchable()
                    ->toggleable()
                    ->icon('heroicon-o-envelope'),

                TextColumn::make('phone')
                    ->label('Phone')
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),

                IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->sortable(),

                TextColumn::make('created_at')
                    ->label('Applied On')
                    ->dateTime('M d, Y')
                    ->sortable()
                    ->toggleable(),
            ])
            ->filters([
                TernaryFilter::make('is_active')
                    ->label('Status')
                    ->trueLabel('Active')
                    ->falseLabel('Inactive'),

                SelectFilter::make('position_applied')
                    ->label('Position')
                    ->options(fn () => Applicant::distinct()->pluck('position_applied', 'position_applied')->toArray()),

                SelectFilter::make('department')
                    ->label('Department')
                    ->options(fn () => Applicant::whereNotNull('department')->distinct()->pluck('department', 'department')->toArray()),
            ])
            ->actions([
                ViewAction::make(),
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListApplicants::route('/'),
            'create' => Pages\CreateApplicant::route('/create'),
            'view' => Pages\ViewApplicant::route('/{record}'),
            'edit' => Pages\EditApplicant::route('/{record}/edit'),
        ];
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::active()->count();
    }

    public static function getNavigationLabel(): string
    {
        return app(TerminologySettings::class)->getApplicantUserLabel(true);
    }

    public static function getModelLabel(): string
    {
        return app(TerminologySettings::class)->getApplicantUserLabel(false);
    }

    public static function getPluralModelLabel(): string
    {
        return app(TerminologySettings::class)->getApplicantUserLabel(true);
    }
}
