<?php

namespace App\Filament\Resources\ExamAttempts;

use App\Enums\ExamAttemptStatus;
use App\Enums\NavigationGroup;
use App\Filament\Resources\ExamAttempts\Pages\ListExamAttempts;
use App\Filament\Resources\ExamAttempts\Pages\ViewExamAttempt;
use App\Models\ExamAttempt;
use App\Settings\TerminologySettings;
use BackedEnum;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Components\RepeatableEntry;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Support\Enums\TextSize;
use Filament\Actions\Action;
use Filament\Actions\BulkAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\ViewAction;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use UnitEnum;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class ExamAttemptResource extends Resource
{
    protected static ?string $model = ExamAttempt::class;

    protected static string|BackedEnum|null $navigationIcon = 'heroicon-o-document-check';

    protected static string|UnitEnum|null $navigationGroup = NavigationGroup::CBT_MANAGEMENT->value;

    protected static ?int $navigationSort = 8;

    protected static ?string $modelLabel = 'Result';

    protected static ?string $pluralModelLabel = 'Exam Results';

    protected static ?string $navigationLabel = 'Results';

    public static function canCreate(): bool
    {
        return false;
    }

    public static function canEdit(\Illuminate\Database\Eloquent\Model $record): bool
    {
        return false;
    }

    public static function infolist(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('Exam Information')
                    ->columns(3)
                    ->schema([
                        TextEntry::make('examSession.exam.title')
                            ->label('Exam'),
                        TextEntry::make('examSession.exam.code')
                            ->label('Code')
                            ->badge(),
                        TextEntry::make('examSession.title')
                            ->label('Session')
                            ->placeholder('—'),
                    ]),

                Section::make('Candidate Information')
                    ->columns(3)
                    ->schema([
                        TextEntry::make('candidate_name')
                            ->label('Name'),
                        TextEntry::make('candidate_identifier')
                            ->label('ID'),
                        TextEntry::make('candidate_type')
                            ->label('Type')
                            ->formatStateUsing(fn (string $state): string => 
                                str_contains($state, 'Student') ? 'Student' : 'Staff'
                            )
                            ->badge(),
                    ]),

                Section::make('Results Summary')
                    ->columns(4)
                    ->schema([
                        TextEntry::make('score_percentage')
                            ->label('Score')
                            ->formatStateUsing(fn ($state): string => number_format($state, 1) . '%')
                            ->size(TextSize::Large)
                            ->color(fn (ExamAttempt $record): string => $record->passed ? 'success' : 'danger'),
                        TextEntry::make('passed')
                            ->label('Result')
                            ->formatStateUsing(fn ($state): string => $state ? 'PASSED' : 'FAILED')
                            ->badge()
                            ->color(fn ($state): string => $state ? 'success' : 'danger')
                            ->size(TextSize::Large),
                        TextEntry::make('score_obtained')
                            ->label('Marks Obtained')
                            ->formatStateUsing(fn (ExamAttempt $record): string => "{$record->score_obtained} / {$record->total_questions}"),
                        TextEntry::make('examSession.exam.passing_score')
                            ->label('Passing Score')
                            ->formatStateUsing(fn ($state): string => $state . '%'),
                    ]),

                Section::make('Answer Statistics')
                    ->columns(5)
                    ->schema([
                        TextEntry::make('total_questions')
                            ->label('Total Questions')
                            ->badge()
                            ->color('gray'),
                        TextEntry::make('correct_answers')
                            ->label('Correct')
                            ->badge()
                            ->color('success'),
                        TextEntry::make('wrong_answers')
                            ->label('Wrong')
                            ->badge()
                            ->color('danger'),
                        TextEntry::make('skipped_questions')
                            ->label('Skipped')
                            ->badge()
                            ->color('warning'),
                        TextEntry::make('answered_questions')
                            ->label('Answered')
                            ->badge()
                            ->color('info'),
                    ]),

                Section::make('Time Information')
                    ->columns(4)
                    ->schema([
                        TextEntry::make('started_at')
                            ->label('Started')
                            ->dateTime('M d, Y H:i:s'),
                        TextEntry::make('completed_at')
                            ->label('Completed')
                            ->dateTime('M d, Y H:i:s')
                            ->placeholder('Not completed'),
                        TextEntry::make('time_spent_seconds')
                            ->label('Time Spent')
                            ->formatStateUsing(function ($state): string {
                                $minutes = floor($state / 60);
                                $seconds = $state % 60;
                                return "{$minutes}m {$seconds}s";
                            }),
                        TextEntry::make('examSession.exam.duration_minutes')
                            ->label('Time Allowed')
                            ->formatStateUsing(fn ($state): string => "{$state} minutes"),
                    ]),

                Section::make('Status & Meta')
                    ->columns(4)
                    ->schema([
                        TextEntry::make('status')
                            ->badge()
                            ->color(fn (ExamAttempt $record): string => $record->status->getColor())
                            ->formatStateUsing(fn (ExamAttempt $record): string => $record->status->getLabel()),
                        TextEntry::make('ip_address')
                            ->label('IP Address')
                            ->placeholder('Not recorded'),
                        TextEntry::make('created_at')
                            ->label('Record Created')
                            ->dateTime(),
                        TextEntry::make('updated_at')
                            ->label('Last Updated')
                            ->dateTime(),
                    ]),

                Section::make('Detailed Answers')
                    ->description('Review individual question responses')
                    ->collapsible()
                    ->collapsed()
                    ->columnSpanFull()
                    ->schema([
                        RepeatableEntry::make('answers')
                            ->schema([
                                TextEntry::make('question_index')
                                    ->label('Q#'),
                                TextEntry::make('question.question_text')
                                    ->label('Question')
                                    ->limit(60),
                                TextEntry::make('selected_answer')
                                    ->label('Selected')
                                    ->placeholder('Skipped'),
                                TextEntry::make('correct_answer')
                                    ->label('Correct'),
                                TextEntry::make('is_correct')
                                    ->label('Result')
                                    ->formatStateUsing(fn ($state) => match ($state) {
                                        true => '✓ Correct',
                                        false => '✗ Wrong',
                                        default => '— Skipped',
                                    })
                                    ->color(fn ($state) => match ($state) {
                                        true => 'success',
                                        false => 'danger',
                                        default => 'warning',
                                    }),
                            ])
                            ->columns(5),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        $terminology = app(TerminologySettings::class);

        return $table
            ->columns([
                TextColumn::make('examSession.exam.code')
                    ->label('Exam')
                    ->searchable()
                    ->sortable()
                    ->badge()
                    ->color('gray'),

                TextColumn::make('examSession.exam.title')
                    ->label('Exam Title')
                    ->searchable()
                    ->limit(25)
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('candidate_name')
                    ->label('Candidate')
                    ->searchable(query: function (Builder $query, string $search): Builder {
                        return $query->whereHas('candidate', function ($q) use ($search) {
                            $q->where('name', 'like', "%{$search}%");
                        });
                    }),

                TextColumn::make('candidate_identifier')
                    ->label('ID')
                    ->searchable(query: function (Builder $query, string $search): Builder {
                        return $query;
                    }),

                TextColumn::make('candidate_type')
                    ->label('Type')
                    ->formatStateUsing(function (string $state): string {
                        $terminology = app(TerminologySettings::class);
                        return str_contains($state, 'Student') 
                            ? $terminology->secondary_user_singular 
                            : $terminology->primary_user_singular;
                    })
                    ->badge()
                    ->color(fn (string $state): string => str_contains($state, 'Student') ? 'info' : 'warning'),

                TextColumn::make('score_percentage')
                    ->label('Score')
                    ->formatStateUsing(fn ($state): string => number_format($state, 1) . '%')
                    ->sortable()
                    ->color(fn (ExamAttempt $record): string => $record->passed ? 'success' : 'danger'),

                TextColumn::make('correct_answers')
                    ->label('Correct')
                    ->formatStateUsing(fn (ExamAttempt $record): string => "{$record->correct_answers}/{$record->total_questions}")
                    ->sortable(),

                TextColumn::make('passed')
                    ->label('Result')
                    ->formatStateUsing(fn ($state): string => $state ? 'PASS' : 'FAIL')
                    ->badge()
                    ->color(fn ($state): string => $state ? 'success' : 'danger'),

                TextColumn::make('status')
                    ->badge()
                    ->color(fn (ExamAttempt $record): string => $record->status->getColor())
                    ->formatStateUsing(fn (ExamAttempt $record): string => $record->status->getLabel()),

                TextColumn::make('time_spent_seconds')
                    ->label('Time Spent')
                    ->formatStateUsing(function ($state): string {
                        $minutes = floor($state / 60);
                        $seconds = $state % 60;
                        return "{$minutes}m {$seconds}s";
                    })
                    ->sortable(),

                TextColumn::make('started_at')
                    ->label('Started')
                    ->dateTime('M d, H:i')
                    ->sortable(),

                TextColumn::make('completed_at')
                    ->label('Completed')
                    ->dateTime('M d, H:i')
                    ->sortable()
                    ->placeholder('—'),
            ])
            ->filters([
                SelectFilter::make('status')
                    ->options(collect(ExamAttemptStatus::cases())->mapWithKeys(
                        fn (ExamAttemptStatus $status) => [$status->value => $status->getLabel()]
                    )),

                SelectFilter::make('passed')
                    ->label('Result')
                    ->options([
                        '1' => 'Passed',
                        '0' => 'Failed',
                    ]),

                SelectFilter::make('exam')
                    ->relationship('examSession.exam', 'title')
                    ->searchable()
                    ->preload(),
            ])
            ->actions([
                ViewAction::make(),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    BulkAction::make('export_excel')
                        ->label('Export to Excel')
                        ->icon('heroicon-o-document-arrow-down')
                        ->deselectRecordsAfterCompletion()
                        ->action(function (Collection $records) {
                            return self::exportToExcel($records);
                        }),
                ]),
            ])
            ->defaultSort('started_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListExamAttempts::route('/'),
            'view' => ViewExamAttempt::route('/{record}'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->with(['examSession.exam', 'candidate']);
    }

    /**
     * Export exam attempts to Excel
     */
    public static function exportToExcel(Collection $records): \Symfony\Component\HttpFoundation\StreamedResponse
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Exam Results');

        // Define headers
        $headers = [
            'A1' => 'Exam Code',
            'B1' => 'Exam Title',
            'C1' => 'Candidate Name',
            'D1' => 'Candidate ID',
            'E1' => 'Candidate Type',
            'F1' => 'Score (%)',
            'G1' => 'Correct Answers',
            'H1' => 'Total Questions',
            'I1' => 'Result',
            'J1' => 'Status',
            'K1' => 'Time Spent',
            'L1' => 'Started At',
            'M1' => 'Completed At',
        ];

        // Style headers
        foreach ($headers as $cell => $header) {
            $sheet->setCellValue($cell, $header);
            $sheet->getStyle($cell)->getFont()->setBold(true);
            $sheet->getStyle($cell)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setRGB('4F46E5');
            $sheet->getStyle($cell)->getFont()->getColor()->setRGB('FFFFFF');
            $sheet->getStyle($cell)->getAlignment()
                ->setHorizontal(Alignment::HORIZONTAL_CENTER);
        }

        // Set column widths
        $sheet->getColumnDimension('A')->setWidth(15);
        $sheet->getColumnDimension('B')->setWidth(30);
        $sheet->getColumnDimension('C')->setWidth(25);
        $sheet->getColumnDimension('D')->setWidth(15);
        $sheet->getColumnDimension('E')->setWidth(15);
        $sheet->getColumnDimension('F')->setWidth(12);
        $sheet->getColumnDimension('G')->setWidth(15);
        $sheet->getColumnDimension('H')->setWidth(15);
        $sheet->getColumnDimension('I')->setWidth(10);
        $sheet->getColumnDimension('J')->setWidth(12);
        $sheet->getColumnDimension('K')->setWidth(12);
        $sheet->getColumnDimension('L')->setWidth(18);
        $sheet->getColumnDimension('M')->setWidth(18);

        // Add data
        $row = 2;
        foreach ($records as $attempt) {
            $minutes = floor($attempt->time_spent_seconds / 60);
            $seconds = $attempt->time_spent_seconds % 60;
            $terminology = app(TerminologySettings::class);
            
            $sheet->setCellValue('A' . $row, $attempt->examSession?->exam?->code ?? '—');
            $sheet->setCellValue('B' . $row, $attempt->examSession?->exam?->title ?? '—');
            $sheet->setCellValue('C' . $row, $attempt->candidate_name);
            $sheet->setCellValue('D' . $row, $attempt->candidate_identifier);
            $sheet->setCellValue('E' . $row, str_contains($attempt->candidate_type, 'Student') 
                ? $terminology->secondary_user_singular 
                : $terminology->primary_user_singular);
            $sheet->setCellValue('F' . $row, number_format($attempt->score_percentage, 1));
            $sheet->setCellValue('G' . $row, $attempt->correct_answers);
            $sheet->setCellValue('H' . $row, $attempt->total_questions);
            $sheet->setCellValue('I' . $row, $attempt->passed ? 'PASS' : 'FAIL');
            $sheet->setCellValue('J' . $row, $attempt->status->getLabel());
            $sheet->setCellValue('K' . $row, "{$minutes}m {$seconds}s");
            $sheet->setCellValue('L' . $row, $attempt->started_at?->format('M d, Y H:i'));
            $sheet->setCellValue('M' . $row, $attempt->completed_at?->format('M d, Y H:i') ?? '—');

            // Color the result cell
            $resultColor = $attempt->passed ? '22C55E' : 'EF4444';
            $sheet->getStyle('I' . $row)->getFont()->getColor()->setRGB($resultColor);
            $sheet->getStyle('I' . $row)->getFont()->setBold(true);

            $row++;
        }

        // Auto-filter
        $sheet->setAutoFilter('A1:M' . ($row - 1));

        // Freeze header row
        $sheet->freezePane('A2');

        $filename = 'exam_results_' . now()->format('Y-m-d_His') . '.xlsx';

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new Xlsx($spreadsheet);
            $writer->save('php://output');
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }
}
