<?php

namespace App\Filament\Resources\ExamSessions;

use App\Enums\ExamSessionStatus;
use App\Enums\NavigationGroup;
use App\Filament\Resources\ExamSessions\Pages\CreateExamSession;
use App\Filament\Resources\ExamSessions\Pages\EditExamSession;
use App\Filament\Resources\ExamSessions\Pages\ListExamSessions;
use App\Filament\Resources\ExamSessions\Pages\ViewExamSession;
use App\Models\Exam;
use App\Models\ExamSession;
use BackedEnum;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Infolists\Components\TextEntry;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Actions\Action;
use Filament\Actions\BulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use UnitEnum;

class ExamSessionResource extends Resource
{
    protected static ?string $model = ExamSession::class;

    protected static string|BackedEnum|null $navigationIcon = 'heroicon-o-calendar-days';

    protected static string|UnitEnum|null $navigationGroup = NavigationGroup::CBT_MANAGEMENT->value;

    protected static ?int $navigationSort = 7;

    protected static ?string $recordTitleAttribute = 'title';

    protected static ?string $modelLabel = 'Exam Session';

    protected static ?string $pluralModelLabel = 'Exam Sessions';

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('Session Details')
                    ->description('Configure the exam session')
                    ->columns(2)
                    ->schema([
                        Select::make('exam_id')
                            ->label('Exam')
                            ->relationship('exam', 'title')
                            ->searchable()
                            ->preload()
                            ->required()
                            ->default(request()->get('exam_id'))
                            ->columnSpanFull(),

                        TextInput::make('title')
                            ->label('Session Title (Optional)')
                            ->maxLength(255)
                            ->placeholder('e.g., Morning Session, Batch A')
                            ->helperText('Give this session a name to distinguish it from others')
                            ->columnSpanFull(),

                        DateTimePicker::make('start_time')
                            ->label('Start Time')
                            ->required()
                            ->native(false)
                            ->seconds(false)
                            ->displayFormat('d M Y - H:i')
                            ->columnSpan(1),

                        DateTimePicker::make('end_time')
                            ->label('End Time')
                            ->required()
                            ->native(false)
                            ->seconds(false)
                            ->displayFormat('d M Y - H:i')
                            ->after('start_time')
                            ->columnSpan(1),
                    ]),

                Section::make('Session Limits')
                    ->description('Set limits for this session')
                    ->columns(2)
                    ->schema([
                        TextInput::make('max_attempts')
                            ->label('Max Attempts per Candidate')
                            ->numeric()
                            ->default(1)
                            ->minValue(1)
                            ->maxValue(10)
                            ->required()
                            ->helperText('How many times can each candidate attempt this exam'),

                        TextInput::make('max_participants')
                            ->label('Max Participants')
                            ->numeric()
                            ->nullable()
                            ->minValue(1)
                            ->placeholder('Unlimited')
                            ->helperText('Leave empty for unlimited participants'),
                    ]),

                Section::make('Access Control')
                    ->description('Restrict access to this session')
                    ->schema([
                        TextInput::make('access_code')
                            ->label('Access Code (Optional)')
                            ->maxLength(20)
                            ->placeholder('Leave empty for no code')
                            ->helperText('Candidates must enter this code to access the exam'),

                        Textarea::make('notes')
                            ->label('Internal Notes')
                            ->rows(2)
                            ->placeholder('Notes for administrators only'),
                    ]),

                Section::make('Status')
                    ->schema([
                        Select::make('status')
                            ->options(collect(ExamSessionStatus::cases())->mapWithKeys(
                                fn (ExamSessionStatus $status) => [$status->value => $status->getLabel()]
                            ))
                            ->default(ExamSessionStatus::SCHEDULED->value)
                            ->required(),
                    ]),
            ]);
    }

    public static function infolist(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('Session Details')
                    ->columns(3)
                    ->schema([
                        TextEntry::make('exam.title')
                            ->label('Exam'),
                        TextEntry::make('exam.code')
                            ->label('Exam Code')
                            ->badge(),
                        TextEntry::make('status')
                            ->badge()
                            ->color(fn (ExamSession $record): string => $record->status->getColor())
                            ->formatStateUsing(fn (ExamSession $record): string => $record->status->getLabel()),
                    ]),

                Section::make('Schedule')
                    ->columns(4)
                    ->schema([
                        TextEntry::make('start_time')
                            ->dateTime('M d, Y H:i'),
                        TextEntry::make('end_time')
                            ->dateTime('M d, Y H:i'),
                        TextEntry::make('max_attempts')
                            ->label('Max Attempts'),
                        TextEntry::make('max_participants')
                            ->label('Max Participants')
                            ->placeholder('Unlimited'),
                    ]),

                Section::make('Statistics')
                    ->columns(4)
                    ->schema([
                        TextEntry::make('participant_count')
                            ->label('Participants')
                            ->badge()
                            ->color('info'),
                        TextEntry::make('completed_count')
                            ->label('Completed')
                            ->badge()
                            ->color('success'),
                        TextEntry::make('average_score')
                            ->label('Avg Score')
                            ->formatStateUsing(fn ($state) => $state ? number_format($state, 1) . '%' : 'N/A'),
                        TextEntry::make('pass_rate')
                            ->label('Pass Rate')
                            ->formatStateUsing(fn ($state) => $state ? number_format($state, 1) . '%' : 'N/A'),
                    ]),

                Section::make('Access')
                    ->columns(2)
                    ->schema([
                        TextEntry::make('access_code')
                            ->label('Access Code')
                            ->placeholder('No access code')
                            ->copyable(),
                        TextEntry::make('notes')
                            ->label('Notes')
                            ->placeholder('No notes'),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('exam.code')
                    ->label('Exam Code')
                    ->searchable()
                    ->sortable()
                    ->badge()
                    ->color('gray'),

                TextColumn::make('exam.title')
                    ->label('Exam')
                    ->searchable()
                    ->sortable()
                    ->limit(30)
                    ->tooltip(fn (ExamSession $record): string => $record->exam->title),

                TextColumn::make('title')
                    ->label('Session')
                    ->placeholder('—')
                    ->searchable(),

                TextColumn::make('start_time')
                    ->label('Start')
                    ->dateTime('M d, Y H:i')
                    ->sortable(),

                TextColumn::make('end_time')
                    ->label('End')
                    ->dateTime('M d, Y H:i')
                    ->sortable(),

                TextColumn::make('status')
                    ->badge()
                    ->color(fn (ExamSession $record): string => $record->status->getColor())
                    ->icon(fn (ExamSession $record): string => $record->status->getIcon())
                    ->formatStateUsing(fn (ExamSession $record): string => $record->status->getLabel()),

                TextColumn::make('attempts_count')
                    ->label('Attempts')
                    ->counts('attempts')
                    ->sortable()
                    ->badge()
                    ->color('info'),

                TextColumn::make('max_attempts')
                    ->label('Max')
                    ->sortable(),

                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('status')
                    ->options(collect(ExamSessionStatus::cases())->mapWithKeys(
                        fn (ExamSessionStatus $status) => [$status->value => $status->getLabel()]
                    )),

                SelectFilter::make('exam_id')
                    ->label('Exam')
                    ->relationship('exam', 'title')
                    ->searchable()
                    ->preload(),
            ])
            ->actions([
                ViewAction::make(),
                EditAction::make(),
                Action::make('activate')
                    ->label('Activate')
                    ->icon('heroicon-o-play')
                    ->color('success')
                    ->requiresConfirmation()
                    ->visible(fn (ExamSession $record): bool => $record->status === ExamSessionStatus::SCHEDULED)
                    ->action(fn (ExamSession $record) => $record->update(['status' => ExamSessionStatus::ACTIVE])),
                Action::make('complete')
                    ->label('Complete')
                    ->icon('heroicon-o-check-circle')
                    ->color('gray')
                    ->requiresConfirmation()
                    ->visible(fn (ExamSession $record): bool => $record->status === ExamSessionStatus::ACTIVE)
                    ->action(fn (ExamSession $record) => $record->update(['status' => ExamSessionStatus::COMPLETED])),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    BulkAction::make('activate')
                        ->label('Activate')
                        ->icon('heroicon-o-play')
                        ->requiresConfirmation()
                        ->action(fn ($records) => $records->each->update(['status' => ExamSessionStatus::ACTIVE])),
                    BulkAction::make('complete')
                        ->label('Mark Completed')
                        ->icon('heroicon-o-check-circle')
                        ->requiresConfirmation()
                        ->action(fn ($records) => $records->each->update(['status' => ExamSessionStatus::COMPLETED])),
                ]),
            ])
            ->defaultSort('start_time', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListExamSessions::route('/'),
            'create' => CreateExamSession::route('/create'),
            'view' => ViewExamSession::route('/{record}'),
            'edit' => EditExamSession::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->with(['exam'])
            ->withCount('attempts');
    }
}
