<?php

namespace App\Filament\Resources\Exams;

use App\Enums\NavigationGroup;
use App\Enums\QuestionSelectionType;
use App\Filament\Resources\Exams\Pages\CreateExam;
use App\Filament\Resources\Exams\Pages\EditExam;
use App\Filament\Resources\Exams\Pages\ListExams;
use App\Filament\Resources\Exams\Pages\ViewExam;
use App\Models\Exam;
use App\Models\QuestionBank;
use App\Models\QuestionCategory;
use App\Settings\TerminologySettings;
use BackedEnum;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\KeyValue;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Radio;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Fieldset;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Actions\Action;
use Filament\Actions\BulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\RestoreAction;
use Filament\Actions\ForceDeleteAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Tables;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use UnitEnum;

class ExamResource extends Resource
{
    protected static ?string $model = Exam::class;

    protected static string|BackedEnum|null $navigationIcon = 'heroicon-o-clipboard-document-list';

    protected static string|UnitEnum|null $navigationGroup = NavigationGroup::CBT_MANAGEMENT->value;

    protected static ?int $navigationSort = 6;

    protected static ?string $recordTitleAttribute = 'title';

    protected static ?string $modelLabel = 'Exam';

    protected static ?string $pluralModelLabel = 'Exams';

    public static function form(Schema $schema): Schema
    {
        $terminology = app(TerminologySettings::class);

        return $schema
            ->components([
                Section::make('Basic Information')
                    ->description('Define the exam details')
                    ->columns(2)
                    ->schema([
                        TextInput::make('title')
                            ->required()
                            ->maxLength(255)
                            ->columnSpan(1),

                        TextInput::make('code')
                            ->maxLength(20)
                            ->unique(ignoreRecord: true)
                            ->placeholder('Auto-generated if empty')
                            ->helperText('Unique exam code for reference')
                            ->columnSpan(1),

                        Textarea::make('description')
                            ->rows(2)
                            ->columnSpanFull(),

                        Textarea::make('instructions')
                            ->rows(3)
                            ->helperText('Instructions shown to candidates before the exam')
                            ->columnSpanFull(),
                    ]),

                Section::make('Exam Configuration')
                    ->description('Set up exam rules and scoring')
                    ->columns(3)
                    ->schema([
                        TextInput::make('duration_minutes')
                            ->label('Duration (minutes)')
                            ->required()
                            ->numeric()
                            ->default(60)
                            ->minValue(1)
                            ->maxValue(480)
                            ->suffix('mins'),

                        TextInput::make('total_questions')
                            ->label('Number of Questions')
                            ->required()
                            ->numeric()
                            ->minValue(1)
                            ->maxValue(500),

                        TextInput::make('passing_score')
                            ->label('Passing Score (%)')
                            ->required()
                            ->numeric()
                            ->default(50)
                            ->minValue(0)
                            ->maxValue(100)
                            ->suffix('%'),

                        TextInput::make('marks_per_question')
                            ->label('Marks per Question')
                            ->required()
                            ->numeric()
                            ->default(1)
                            ->minValue(0.1)
                            ->step(0.1)
                            ->helperText('Can be decimal e.g. 0.5'),

                        TextInput::make('negative_marking')
                            ->label('Negative Marking')
                            ->numeric()
                            ->default(0)
                            ->minValue(0)
                            ->step(0.1)
                            ->helperText('Points deducted for wrong answers'),

                        Placeholder::make('total_marks')
                            ->label('Total Marks')
                            ->content(fn (Get $get): string => 
                                number_format((float) ($get('total_questions') ?? 0) * (float) ($get('marks_per_question') ?? 1), 2) . ' marks'
                            ),
                    ]),

                Section::make('Question Selection')
                    ->description('Choose how questions will be selected for this exam')
                    ->schema([
                        Radio::make('question_type')
                            ->label('Question Type')
                            ->options(Exam::QUESTION_TYPES)
                            ->default('mcq')
                            ->required()
                            ->live()
                            ->afterStateUpdated(function ($set) {
                                $set('question_ids', null);
                                $set('category_distribution', null);
                            })
                            ->helperText('MCQ exams are auto-graded. Essay exams require manual grading by examiners.'),

                        Radio::make('question_selection_type')
                            ->label('Selection Method')
                            ->options([
                                'category_based' => 'Subject/Category-Based - Select all questions from chosen subjects',
                                'manual' => 'Manual Selection - Choose specific questions',
                            ])
                            ->default('category_based')
                            ->required()
                            ->live()
                            ->afterStateUpdated(fn ($set) => $set('question_ids', null)),

                        // Subject/Category-based selection - select subjects, all questions from those subjects get added
                        Select::make('category_distribution')
                            ->label('Select Subjects/Categories')
                            ->multiple()
                            ->searchable()
                            ->preload()
                            ->options(function (Get $get) {
                                $questionType = $get('question_type') ?? 'mcq';
                                $targetAudience = $get('target_audience') ?? 'student';
                                
                                // Get categories that have questions of the selected type AND target audience
                                // MCQ includes: single_select, true_false, multiple_select
                                // Essay is just: essay
                                return QuestionCategory::active()
                                    ->whereHas('questions', function ($query) use ($questionType, $targetAudience) {
                                        $query->where('is_active', true);
                                        
                                        // Filter by question type
                                        if ($questionType === 'essay') {
                                            $query->where('question_type', 'essay');
                                        } else {
                                            $query->whereIn('question_type', ['single_select', 'true_false', 'multiple_select']);
                                        }
                                        
                                        // Filter by target audience
                                        if (in_array($targetAudience, ['student', 'staff', 'applicant'])) {
                                            $query->where('target_audience', $targetAudience);
                                        }
                                        // 'both' = student + staff, 'all' = everyone - no filter needed
                                    })
                                    ->pluck('name', 'id');
                            })
                            ->visible(fn (Get $get) => $get('question_selection_type') === 'category_based')
                            ->helperText(fn (Get $get) => $get('question_type') === 'essay' 
                                ? 'All essay questions from selected subjects matching the target audience will be included' 
                                : 'All MCQ questions from selected subjects matching the target audience will be included')
                            ->live()
                            ->afterStateUpdated(function ($state, $set, Get $get) {
                                if ($state) {
                                    $questionType = $get('question_type') ?? 'mcq';
                                    $targetAudience = $get('target_audience') ?? 'student';
                                    
                                    $query = QuestionBank::whereIn('category_id', $state)
                                        ->where('is_active', true);
                                    
                                    // Filter by question type
                                    if ($questionType === 'essay') {
                                        $query->where('question_type', 'essay');
                                    } else {
                                        $query->whereIn('question_type', ['single_select', 'true_false', 'multiple_select']);
                                    }
                                    
                                    // Filter by target audience
                                    if (in_array($targetAudience, ['student', 'staff', 'applicant'])) {
                                        $query->where('target_audience', $targetAudience);
                                    }
                                    
                                    $set('total_questions', $query->count());
                                }
                            }),

                        // Manual selection - filtered by question type
                        Select::make('question_ids')
                            ->label('Select Questions')
                            ->multiple()
                            ->searchable()
                            ->preload()
                            ->options(function (Get $get) {
                                $questionType = $get('question_type') ?? 'mcq';
                                
                                $query = QuestionBank::active();
                                
                                // MCQ includes: single_select, true_false, multiple_select
                                // Essay is just: essay
                                if ($questionType === 'essay') {
                                    $query->where('question_type', 'essay');
                                } else {
                                    $query->whereIn('question_type', ['single_select', 'true_false', 'multiple_select']);
                                }
                                
                                return $query->limit(500)
                                    ->get()
                                    ->mapWithKeys(fn ($q) => [
                                        $q->id => ($q->category?->name ?? 'No Category') . ' - ' . substr(strip_tags($q->question_text), 0, 80) . '...'
                                    ]);
                            })
                            ->visible(fn (Get $get) => $get('question_selection_type') === 'manual')
                            ->helperText(fn (Get $get) => $get('question_type') === 'essay'
                                ? 'Select essay questions for this exam'
                                : 'Select MCQ questions for this exam'),
                    ]),

                Section::make('Target Audience')
                    ->description('Who can take this exam')
                    ->columns(3)
                    ->schema([
                        Select::make('target_audience')
                            ->label('Audience')
                            ->options(Exam::getTargetAudienceOptions())
                            ->required()
                            ->default('student')
                            ->live()
                            ->afterStateUpdated(function ($set) {
                                // Clear category selection when audience changes so user re-selects with correct filter
                                $set('category_distribution', null);
                                $set('total_questions', 0);
                            }),

                        TextInput::make('target_period')
                            ->label($terminology->period_label)
                            ->placeholder("e.g., 2024/2025"),

                        TextInput::make('target_department')
                            ->label($terminology->division_label)
                            ->placeholder("e.g., Engineering"),
                    ]),

                Section::make('Examiner Assignment')
                    ->description('Assign staff members to grade and manage this exam')
                    ->schema([
                        Select::make('examiners')
                            ->label('Assigned Examiners')
                            ->multiple()
                            ->searchable()
                            ->preload()
                            ->relationship(
                                name: 'examiners',
                                titleAttribute: 'full_name',
                                modifyQueryUsing: fn ($query) => $query->active()
                            )
                            ->getOptionLabelFromRecordUsing(fn ($record) => "{$record->full_name} ({$record->staff_number}) - {$record->position}")
                            ->helperText('Select staff members who will grade essay answers and manage exam results. Only staff with the Examiner role can grade exams.')
                            ->columnSpanFull(),
                    ]),

                Section::make('Exam Behavior')
                    ->description('These settings OVERRIDE the global defaults from Settings > Exam Settings for this specific exam.')
                    ->columns(2)
                    ->schema([
                        Checkbox::make('shuffle_questions')
                            ->label('Shuffle Questions')
                            ->default(true)
                            ->helperText('Randomize question order for each attempt'),

                        Checkbox::make('shuffle_options')
                            ->label('Shuffle Options')
                            ->default(true)
                            ->helperText('Randomize answer options for each question'),

                        Checkbox::make('show_result_immediately')
                            ->label('Show Result Immediately')
                            ->default(true)
                            ->helperText('Show score right after exam completion'),

                        Checkbox::make('allow_review')
                            ->label('Allow Review Before Submit')
                            ->default(true)
                            ->helperText('Allow candidates to review answers before submitting'),

                        Checkbox::make('show_correct_answers')
                            ->label('Show Correct Answers')
                            ->default(false)
                            ->helperText('Reveal correct answers after completion'),

                        Checkbox::make('is_active')
                            ->label('Active')
                            ->default(true)
                            ->helperText('Inactive exams cannot have new sessions'),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        $terminology = app(TerminologySettings::class);

        return $table
            ->columns([
                TextColumn::make('code')
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->badge()
                    ->color('gray'),

                TextColumn::make('title')
                    ->searchable()
                    ->sortable()
                    ->limit(40)
                    ->tooltip(fn (Exam $record): string => $record->title),

                TextColumn::make('question_type')
                    ->label('Type')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'essay' => 'warning',
                        default => 'info',
                    })
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'essay' => 'Essay',
                        default => 'MCQ',
                    }),

                TextColumn::make('total_questions')
                    ->label('Questions')
                    ->sortable()
                    ->badge(),

                TextColumn::make('duration_minutes')
                    ->label('Duration')
                    ->suffix(' mins')
                    ->sortable(),

                TextColumn::make('passing_score')
                    ->label('Pass %')
                    ->suffix('%')
                    ->sortable(),

                TextColumn::make('target_audience')
                    ->label('Audience')
                    ->formatStateUsing(function (string $state): string {
                        $terminology = app(TerminologySettings::class);
                        return match ($state) {
                            'student' => $terminology->primary_user_plural,
                            'staff' => $terminology->secondary_user_plural,
                            'both' => 'Both',
                            default => $state,
                        };
                    })
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'student' => 'info',
                        'staff' => 'warning',
                        'both' => 'success',
                        default => 'gray',
                    }),

                TextColumn::make('sessions_count')
                    ->label('Sessions')
                    ->counts('sessions')
                    ->sortable()
                    ->badge()
                    ->color('gray'),

                IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->sortable(),

                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                TernaryFilter::make('is_active')
                    ->label('Active Status'),

                SelectFilter::make('target_audience')
                    ->label('Target Audience')
                    ->options(Exam::getTargetAudienceOptions()),

                SelectFilter::make('question_selection_type')
                    ->label('Selection Type')
                    ->options([
                        'category_based' => 'Subject/Category-Based',
                        'manual' => 'Manual',
                    ]),
            ])
            ->actions([
                ViewAction::make(),
                EditAction::make(),
                Action::make('create_session')
                    ->label('Create Session')
                    ->icon('heroicon-o-calendar-days')
                    ->color('success')
                    ->url(fn (Exam $record): string => 
                        route('filament.admin.resources.exam-sessions.create', ['exam_id' => $record->id])
                    ),
                DeleteAction::make(),
                RestoreAction::make(),
                ForceDeleteAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    BulkAction::make('activate')
                        ->label('Activate')
                        ->icon('heroicon-o-check-circle')
                        ->requiresConfirmation()
                        ->action(fn ($records) => $records->each->update(['is_active' => true])),
                    BulkAction::make('deactivate')
                        ->label('Deactivate')
                        ->icon('heroicon-o-x-circle')
                        ->color('warning')
                        ->requiresConfirmation()
                        ->action(fn ($records) => $records->each->update(['is_active' => false])),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListExams::route('/'),
            'create' => CreateExam::route('/create'),
            'view' => ViewExam::route('/{record}'),
            'edit' => EditExam::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
