<?php

namespace App\Filament\Resources\QuestionBanks\Schemas;

use App\Models\QuestionBank;
use App\Models\QuestionCategory;
use App\Settings\TerminologySettings;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Fieldset;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;

class QuestionBankForm
{
    public static function configure(Schema $schema): Schema
    {
        $terminology = app(TerminologySettings::class);
        
        return $schema
            ->columns(2)
            ->components([
                Section::make('Question Details')
                    ->description('Basic information about the question')
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                Select::make('question_type')
                                    ->label('Question Type')
                                    ->options(QuestionBank::getQuestionTypeOptions())
                                    ->required()
                                    ->default('single_select')
                                    ->live()
                                    ->columnSpan(1),
                                
                                Select::make('target_audience')
                                    ->label('Target Audience')
                                    ->options(QuestionBank::getTargetAudienceOptions())
                                    ->required()
                                    ->default('student')
                                    ->live()
                                    ->columnSpan(1),
                            ]),
                        
                        RichEditor::make('question_text')
                            ->label('Question Text')
                            ->required()
                            ->toolbarButtons([
                                'bold',
                                'italic',
                                'underline',
                                'bulletList',
                                'orderedList',
                            ])
                            ->columnSpanFull(),
                        
                        Grid::make(4)
                            ->schema([
                                Select::make('category_id')
                                    ->label('Subject')
                                    ->options(
                                        QuestionCategory::where('is_active', true)
                                            ->orderBy('name')
                                            ->pluck('name', 'id')
                                    )
                                    ->searchable()
                                    ->preload()
                                    ->required()
                                    ->createOptionForm([
                                        TextInput::make('name')
                                            ->label('Subject Name')
                                            ->required()
                                            ->maxLength(255),
                                        Textarea::make('description')
                                            ->rows(2),
                                    ])
                                    ->createOptionUsing(function (array $data): int {
                                        return QuestionCategory::create([
                                            'name' => $data['name'],
                                            'description' => $data['description'] ?? null,
                                            'is_active' => true,
                                        ])->id;
                                    })
                                    ->helperText('Select or create a subject for this question'),
                                
                                TextInput::make('session')
                                    ->label($terminology->period_label)
                                    ->placeholder('e.g., 2024/2025')
                                    ->visible(fn (Get $get): bool => $get('target_audience') === 'student'),

                                TextInput::make('department')
                                    ->label($terminology->division_label)
                                    ->placeholder('Enter ' . strtolower($terminology->division_label)),
                            ]),
                    ])
                    ->columnSpan(2),

                Section::make('Answer Options')
                    ->description('Configure the answer options and correct answer')
                    ->schema([
                        Repeater::make('options')
                            ->label('Answer Options')
                            ->schema([
                                TextInput::make('key')
                                    ->label('Option Key')
                                    ->placeholder('A, B, C, D...')
                                    ->required()
                                    ->maxLength(5),
                                
                                TextInput::make('text')
                                    ->label('Option Text')
                                    ->required()
                                    ->maxLength(500),
                            ])
                            ->reorderable()
                            ->collapsible()
                            ->itemLabel(fn (array $state): ?string => $state['key'] ?? null)
                            ->minItems(2)
                            ->maxItems(6)
                            ->default([
                                ['key' => 'A', 'text' => ''],
                                ['key' => 'B', 'text' => ''],
                            ])
                            ->live()
                            ->visible(fn (Get $get): bool => $get('question_type') === 'single_select'),
                        
                        Select::make('correct_answer')
                            ->label('Correct Answer')
                            ->options([
                                'A' => 'True',
                                'B' => 'False',
                            ])
                            ->required()
                            ->visible(fn (Get $get): bool => $get('question_type') === 'true_false'),
                        
                        Select::make('correct_answer')
                            ->label('Correct Answer Key')
                            ->options(function (Get $get) {
                                $options = collect($get('options') ?? [])
                                    ->filter(fn ($option) => !empty($option['key']) && !empty($option['text']))
                                    ->mapWithKeys(fn ($option) => [$option['key'] => $option['text'] ?: $option['key']])
                                    ->toArray();
                                
                                return $options;
                            })
                            ->required()
                            ->visible(fn (Get $get): bool => $get('question_type') === 'single_select'),
                        
                        Textarea::make('explanation')
                            ->label('Answer Explanation')
                            ->placeholder('Optional explanation for the correct answer...')
                            ->rows(3)
                            ->columnSpanFull(),
                    ])
                    ->columnSpan(2)
                    ->visible(fn (Get $get): bool => in_array($get('question_type'), ['single_select', 'true_false'])),

                Section::make('Essay Answer Configuration')
                    ->description('Configure word limits for essay questions')
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                TextInput::make('min_words')
                                    ->label('Minimum Words')
                                    ->numeric()
                                    ->placeholder('Optional')
                                    ->minValue(1)
                                    ->maxValue(10000)
                                    ->helperText('Minimum word count required'),

                                TextInput::make('max_words')
                                    ->label('Maximum Words')
                                    ->numeric()
                                    ->placeholder('Optional')
                                    ->minValue(1)
                                    ->maxValue(10000)
                                    ->helperText('Maximum word count allowed'),
                            ]),
                    ])
                    ->columnSpan(2)
                    ->visible(fn (Get $get): bool => $get('question_type') === 'essay'),

                Section::make('Question Settings')
                    ->description('Additional configuration options')
                    ->schema([
                        Grid::make(3)
                            ->schema([
                                TextInput::make('time_limit')
                                    ->label('Time Limit')
                                    ->numeric()
                                    ->placeholder('Optional')
                                    ->minValue(1)
                                    ->maxValue(3600)
                                    ->suffix('seconds'),

                                TextInput::make('reference')
                                    ->label('Reference/Source')
                                    ->placeholder('Optional reference material or source'),

                                Toggle::make('is_active')
                                    ->label('Active')
                                    ->helperText('Only active questions can be used in exams')
                                    ->default(true),
                            ]),
                    ])
                    ->columnSpan(2)
                    ->hidden(),

                Fieldset::make('Import Information')
                    ->schema([
                        TextInput::make('import_batch_id')
                            ->label('Import Batch ID')
                            ->disabled()
                            ->dehydrated(false),
                        
                        DateTimePicker::make('imported_at')
                            ->label('Imported At')
                            ->disabled()
                            ->dehydrated(false),
                    ])
                    ->columns(2)
                    ->columnSpan(2)
                    ->visible(fn ($record) => $record && $record->import_batch_id),
            ]);
    }
}
