<?php

namespace App\Filament\Resources\Staff\Pages;

use App\Filament\Resources\Staff\StaffResource;
use App\Models\Staff;
use Filament\Actions\Action;
use Filament\Actions\CreateAction;
use Filament\Forms\Components\FileUpload;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ListRecords;
use Illuminate\Support\Facades\Storage;
use League\Csv\Reader;
use League\Csv\Statement;

class ListStaff extends ListRecords
{
    protected static string $resource = StaffResource::class;

    protected function getHeaderActions(): array
    {
        return [
            CreateAction::make(),
            
            Action::make('download_template')
                ->label('Download Template')
                ->icon('heroicon-o-document-arrow-down')
                ->color('gray')
                ->url(route('staff.template'))
                ->openUrlInNewTab(),
            
            Action::make('bulk_import')
                ->label('Bulk Import Staff')
                ->icon('heroicon-o-document-arrow-up')
                ->color('success')
                ->form([
                    FileUpload::make('file')
                        ->label('Import File')
                        ->acceptedFileTypes(['text/csv', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'])
                        ->required()
                        ->helperText('Upload a CSV or Excel file with staff_number, full_name, email, and staff_type as required fields. Download the template below for the correct format.'),
                ])
                ->action(function (array $data) {
                    $this->importStaff($data['file']);
                }),
        ];
    }

    protected function importStaff(string $file): void
    {
        try {
            $path = Storage::disk('local')->path($file);
            $csv = Reader::createFromPath($path, 'r');
            $csv->setHeaderOffset(0);

            $records = Statement::create()->process($csv);
            $successCount = 0;
            $errors = [];

            foreach ($records as $offset => $record) {
                try {
                    // Validate required fields
                    if (empty($record['staff_number']) || empty($record['full_name']) || empty($record['email']) || empty($record['staff_type'])) {
                        $errors[] = "Row " . ($offset + 2) . ": Missing required fields (staff_number, full_name, email, staff_type)";
                        continue;
                    }

                    // Check if staff number already exists
                    if (Staff::where('staff_number', $record['staff_number'])->exists()) {
                        $errors[] = "Row " . ($offset + 2) . ": Staff number already exists: " . $record['staff_number'];
                        continue;
                    }

                    // Validate staff type
                    if (!in_array(strtolower($record['staff_type']), ['senior', 'junior'])) {
                        $errors[] = "Row " . ($offset + 2) . ": Invalid staff_type. Must be 'senior' or 'junior'";
                        continue;
                    }

                    // Validate staff number matches staff type
                    $expectedType = Staff::getStaffTypeFromNumber($record['staff_number']);
                    if ($expectedType !== strtolower($record['staff_type'])) {
                        $errors[] = "Row " . ($offset + 2) . ": Staff number doesn't match staff type. PN### for senior, JN### for junior";
                        continue;
                    }

                    // Check if email already exists
                    if (Staff::where('email', $record['email'])->exists()) {
                        $errors[] = "Row " . ($offset + 2) . ": Email already exists: " . $record['email'];
                        continue;
                    }

                    // Create staff member
                    Staff::create([
                        'staff_number' => $record['staff_number'],
                        'full_name' => $record['full_name'],
                        'email' => $record['email'],
                        'phone' => $record['phone'] ?? null,
                        'department' => $record['department'] ?? 'General',
                        'position' => $record['position'] ?? 'Staff',
                        'staff_type' => strtolower($record['staff_type']),
                        'employment_status' => $record['employment_status'] ?? 'active',
                        'password' => bcrypt($record['password'] ?? 'password123'),
                    ]);

                    $successCount++;
                } catch (\Exception $e) {
                    $errors[] = "Row " . ($offset + 2) . ": " . $e->getMessage();
                }
            }

            // Clean up uploaded file
            Storage::disk('public')->delete($file);

            // Show results
            if ($successCount > 0) {
                Notification::make()
                    ->title('Import Successful')
                    ->body("Successfully imported {$successCount} staff members.")
                    ->success()
                    ->send();
            }

            if (!empty($errors)) {
                $errorMessage = "Import completed with errors:\n" . implode("\n", array_slice($errors, 0, 5));
                if (count($errors) > 5) {
                    $errorMessage .= "\n... and " . (count($errors) - 5) . " more errors.";
                }

                Notification::make()
                    ->title('Import Warnings')
                    ->body($errorMessage)
                    ->warning()
                    ->send();
            }

        } catch (\Exception $e) {
            Notification::make()
                ->title('Import Failed')
                ->body('Error processing file: ' . $e->getMessage())
                ->danger()
                ->send();
        }
    }

    public function getMaxContentWidth(): ?string
    {
        return 'full';
    }
}
