<?php

namespace App\Filament\Resources\Staff\Schemas;

use App\Filament\Resources\Staff\Schemas\Components\DepartmentInput;
use App\Filament\Resources\Staff\Schemas\Components\EmploymentStatusSelect;
use App\Filament\Resources\Staff\Schemas\Components\StaffNumberInput;
use App\Filament\Resources\Staff\Schemas\Components\StaffTypeSelect;
use App\Models\Staff;
use App\Settings\TerminologySettings;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Schema;
use Filament\Schemas\Components\Section;
use Spatie\Permission\Models\Role;

class StaffForm
{
    public static function configure(Schema $schema): Schema
    {
        $terminology = app(TerminologySettings::class);
        
        return $schema
            ->components([
                Section::make($terminology->secondary_user_singular . ' Information')
                    ->description('Enter the ' . strtolower($terminology->secondary_user_singular) . ' details for system access')
                    ->schema([
                        StaffNumberInput::make(),

                        TextInput::make('full_name')
                            ->label('Full Name')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('e.g., Dr. John Doe Smith')
                            ->columnSpan(2),

                        StaffTypeSelect::make(),

                        TextInput::make('email')
                            ->label('Email Address')
                            ->email()
                            ->unique(Staff::class, 'email', ignoreRecord: true)
                            ->maxLength(255)
                            ->required()
                            ->placeholder('e.g., user@example.com')
                            ->helperText('Used for system notifications')
                            ->suffixIcon('heroicon-m-envelope'),

                        TextInput::make('phone')
                            ->label('Phone Number')
                            ->tel()
                            ->maxLength(20)
                            ->placeholder('e.g., +234 802 123 4567')
                            ->suffixIcon('heroicon-m-phone'),

                        DepartmentInput::make(),

                        TextInput::make('position')
                            ->label('Position/Job Title')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('e.g., Manager, Team Lead, Supervisor')
                            ->columnSpan(2),

                        EmploymentStatusSelect::make(),

                        DatePicker::make('termination_date')
                            ->label('Termination Date')
                            ->visible(fn (callable $get) => in_array($get('employment_status'), ['terminated']))
                            ->afterOrEqual(now())
                            ->helperText('Required for terminated ' . strtolower($terminology->secondary_user_plural))
                            ->suffixIcon('heroicon-m-calendar'),

                        TextInput::make('password')
                            ->label('Password')
                            ->password()
                            ->required(fn (string $context): bool => $context === 'create')
                            ->minLength(6)
                            ->dehydrateStateUsing(fn (?string $state): ?string => 
                                filled($state) ? bcrypt($state) : null
                            )
                            ->dehydrated(fn (?string $state): bool => filled($state))
                            ->placeholder('Minimum 6 characters')
                            ->helperText($terminology->secondary_user_singular . ' will use this to login to the system')
                            ->suffixIcon('heroicon-m-key')
                            ->columnSpanFull(),
                    ])
                    ->columns(3)
                    ->columnSpanFull()
                    ->collapsible()
                    ->persistCollapsed(false),

                Section::make('Roles & Permissions')
                    ->description('Assign roles to determine what this ' . strtolower($terminology->secondary_user_singular) . ' can access and manage')
                    ->schema([
                        Select::make('roles')
                            ->label('Assigned Roles')
                            ->multiple()
                            ->searchable()
                            ->preload()
                            ->relationship(
                                name: 'roles',
                                titleAttribute: 'name',
                                modifyQueryUsing: fn ($query) => $query->where('guard_name', 'staff')
                            )
                            ->helperText('Assign the Examiner role for full system access (manage exams, questions, sessions, results, and users)')
                            ->columnSpanFull(),
                    ])
                    ->columns(1)
                    ->columnSpanFull()
                    ->collapsible()
                    ->collapsed(),
            ])
            ->columns(1); // Make the entire schema single column for full width
    }
}
