<?php

namespace App\Filament\Resources\Staff\Tables;

use App\Models\Staff;
use App\Settings\TerminologySettings;
use Filament\Actions\Action;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Tables\Columns\BadgeColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;

class StaffTable
{
    public static function configure(Table $table): Table
    {
        $terminology = app(TerminologySettings::class);
        
        return $table
            ->columns([
                TextColumn::make('staff_number')
                    ->label($terminology->secondary_user_id_label)
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->weight('bold')
                    ->color('primary'),

                TextColumn::make('full_name')
                    ->label('Full Name')
                    ->searchable()
                    ->sortable()
                    ->limit(30)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();
                        if (strlen($state) <= 30) {
                            return null;
                        }
                        return $state;
                    }),

                BadgeColumn::make('staff_type')
                    ->label('Type')
                    ->colors([
                        'primary' => 'senior',
                        'warning' => 'junior',
                    ])
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'senior' => 'Senior',
                        'junior' => 'Junior',
                        default => $state,
                    }),

                TextColumn::make('department')
                    ->label($terminology->division_label)
                    ->searchable()
                    ->sortable()
                    ->limit(20)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();
                        if (strlen($state) <= 20) {
                            return null;
                        }
                        return $state;
                    }),

                TextColumn::make('position')
                    ->label('Position')
                    ->searchable()
                    ->limit(25)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();
                        if (strlen($state) <= 25) {
                            return null;
                        }
                        return $state;
                    }),

                BadgeColumn::make('employment_status')
                    ->label('Status')
                    ->colors([
                        'success' => 'active',
                        'warning' => 'inactive',
                        'danger' => ['suspended', 'terminated'],
                    ])
                    ->formatStateUsing(fn (string $state): string => ucfirst($state)),

                BadgeColumn::make('roles.name')
                    ->label('Role')
                    ->placeholder('No Role')
                    ->badge()
                    ->color(fn ($state): string => $state === 'Examiner' ? 'success' : 'gray')
                    ->icon(fn ($state): string => $state === 'Examiner' ? 'heroicon-m-shield-check' : 'heroicon-m-user')
                    ->sortable(),

                TextColumn::make('email')
                    ->label('Email')
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->copyable(),

                TextColumn::make('phone')
                    ->label('Phone')
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->copyable(),

                TextColumn::make('created_at')
                    ->label('Created')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('staff_type')
                    ->label('Staff Type')
                    ->options([
                        'senior' => 'Senior Staff (PN)',
                        'junior' => 'Junior Staff (JN)',
                    ]),

                SelectFilter::make('employment_status')
                    ->label('Employment Status')
                    ->options([
                        'active' => 'Active',
                        'inactive' => 'Inactive',
                        'suspended' => 'Suspended',
                        'terminated' => 'Terminated',
                    ])
                    ->default('active'),

                SelectFilter::make('department')
                    ->label('Department')
                    ->options(function () {
                        return Staff::query()
                            ->distinct()
                            ->pluck('department', 'department')
                            ->filter()
                            ->sort()
                            ->toArray();
                    }),
            ])
            ->recordActions([
                ViewAction::make(),
                EditAction::make(),
                Action::make('toggle_status')
                    ->label(fn (Staff $record) => $record->employment_status === 'active' ? 'Deactivate' : 'Activate')
                    ->icon(fn (Staff $record) => $record->employment_status === 'active' ? 'heroicon-o-x-circle' : 'heroicon-o-check-circle')
                    ->color(fn (Staff $record) => $record->employment_status === 'active' ? 'danger' : 'success')
                    ->action(function (Staff $record) {
                        $record->update([
                            'employment_status' => $record->employment_status === 'active' ? 'inactive' : 'active'
                        ]);
                    })
                    ->requiresConfirmation()
                    ->modalDescription(fn (Staff $record) => 
                        $record->employment_status === 'active' 
                            ? 'Are you sure you want to deactivate this staff member?' 
                            : 'Are you sure you want to activate this staff member?'
                    ),
                DeleteAction::make(),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    \Filament\Actions\BulkAction::make('activate')
                        ->label('Activate Selected')
                        ->icon('heroicon-m-check-circle')
                        ->color('success')
                        ->requiresConfirmation()
                        ->action(function ($records) use ($terminology) {
                            $count = $records->count();
                            $records->each(fn ($record) => $record->update(['employment_status' => 'active']));
                            
                            \Filament\Notifications\Notification::make()
                                ->title("{$count} " . $terminology->secondary_user_plural . ' activated')
                                ->success()
                                ->send();
                        })
                        ->deselectRecordsAfterCompletion(),
                    
                    \Filament\Actions\BulkAction::make('deactivate')
                        ->label('Deactivate Selected')
                        ->icon('heroicon-m-x-circle')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->action(function ($records) use ($terminology) {
                            $count = $records->count();
                            $records->each(fn ($record) => $record->update(['employment_status' => 'inactive']));
                            
                            \Filament\Notifications\Notification::make()
                                ->title("{$count} " . $terminology->secondary_user_plural . ' deactivated')
                                ->warning()
                                ->send();
                        })
                        ->deselectRecordsAfterCompletion(),
                    
                    DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('staff_number', 'asc')
            ->striped()
            ->paginated([10, 25, 50, 100]);
    }
}
