<?php

namespace App\Filament\Resources\Students\Pages;

use App\Filament\Resources\Students\StudentResource;
use App\Models\Student;
use Filament\Actions\Action;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\Page;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class BulkUpload extends Page implements HasForms
{
    use InteractsWithForms;

    protected static string $resource = StudentResource::class;

    protected string $view = 'filament.resources.students.pages.bulk-upload';

    protected static ?string $navigationLabel = 'Bulk Upload';

    protected static ?string $title = 'Bulk Student Upload';

    public ?array $data = [];

    public function mount(): void
    {
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                FileUpload::make('file')
                    ->label('Upload File')
                    ->acceptedFileTypes(['text/csv', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'])
                    ->maxSize(2048)
                    ->required()
                    ->helperText('Upload a CSV or Excel file with columns: full_name, email, matriculation_number, session (optional), password (optional)')
            ])
            ->statePath('data');
    }

    public function upload(): void
    {
        $data = $this->form->getState();
        
        if (!isset($data['file'])) {
            Notification::make()
                ->title('Error')
                ->body('Please select a file to upload.')
                ->danger()
                ->send();
            return;
        }

        try {
            $file = $data['file'];
            $filePath = storage_path('app/public/' . $file);
            
            $excelData = Excel::toArray([], $filePath)[0];
            
            if (empty($excelData)) {
                Notification::make()
                    ->title('Error')
                    ->body('The uploaded file is empty.')
                    ->danger()
                    ->send();
                return;
            }

            // Get headers from first row
            $headers = array_shift($excelData);
            $headers = array_map('trim', $headers);

            // Validate required headers
            $requiredHeaders = ['full_name', 'email', 'matriculation_number'];
            $missingHeaders = array_diff($requiredHeaders, $headers);
            
            if (!empty($missingHeaders)) {
                Notification::make()
                    ->title('Error')
                    ->body('Missing required columns: ' . implode(', ', $missingHeaders))
                    ->danger()
                    ->send();
                return;
            }

            $successCount = 0;
            $errorCount = 0;
            $errors = [];

            foreach ($excelData as $index => $row) {
                $rowNumber = $index + 2; // +2 because we removed header and arrays are 0-indexed
                
                // Create associative array from row data
                $studentData = array_combine($headers, $row);
                
                // Validate row data
                $validator = Validator::make($studentData, [
                    'full_name' => ['required', 'string', 'max:255'],
                    'email' => ['required', 'email', Rule::unique('students', 'email')],
                    'matriculation_number' => ['required', 'string', Rule::unique('students', 'matriculation_number')],
                    'session' => ['nullable', 'string', 'max:255'],
                    'password' => ['nullable', 'string', 'min:6'],
                ]);

                if ($validator->fails()) {
                    $errorCount++;
                    $errors[] = "Row {$rowNumber}: " . implode(', ', $validator->errors()->all());
                    continue;
                }

                try {
                    Student::create([
                        'full_name' => trim($studentData['full_name']),
                        'email' => trim($studentData['email']),
                        'matriculation_number' => trim($studentData['matriculation_number']),
                        'session' => isset($studentData['session']) ? trim($studentData['session']) : null,
                        'password' => isset($studentData['password']) && !empty($studentData['password']) 
                            ? $studentData['password'] 
                            : $studentData['matriculation_number'], // Default password is the matriculation number
                        'is_active' => true,
                    ]);
                    $successCount++;
                } catch (\Exception $e) {
                    $errorCount++;
                    $errors[] = "Row {$rowNumber}: Failed to create student - " . $e->getMessage();
                }
            }

            // Show results
            if ($successCount > 0) {
                Notification::make()
                    ->title('Success')
                    ->body("Successfully imported {$successCount} students.")
                    ->success()
                    ->send();
            }

            if ($errorCount > 0) {
                $errorMessage = "Failed to import {$errorCount} students.";
                if (count($errors) <= 5) {
                    $errorMessage .= "\n\nErrors:\n" . implode("\n", $errors);
                } else {
                    $errorMessage .= "\n\nShowing first 5 errors:\n" . implode("\n", array_slice($errors, 0, 5));
                    $errorMessage .= "\n... and " . (count($errors) - 5) . " more errors.";
                }
                
                Notification::make()
                    ->title('Import Warnings')
                    ->body($errorMessage)
                    ->warning()
                    ->send();
            }

            // Reset form
            $this->form->fill();

            // Redirect to students list if successful
            if ($successCount > 0 && $errorCount === 0) {
                $this->redirect(route('filament.admin.resources.students.index'));
            }

        } catch (\Exception $e) {
            Notification::make()
                ->title('Error')
                ->body('Failed to process file: ' . $e->getMessage())
                ->danger()
                ->send();
        }
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('upload')
                ->label('Upload Students')
                ->action('upload')
                ->color('primary'),
        ];
    }
}
