<?php

namespace App\Filament\Resources\Students\Pages;

use App\Filament\Resources\Students\StudentResource;
use App\Models\Student;
use Filament\Actions\CreateAction;
use Filament\Actions\Action;
use Filament\Forms\Components\FileUpload;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ListRecords;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class ListStudents extends ListRecords
{
    protected static string $resource = StudentResource::class;

    protected function getHeaderActions(): array
    {
        return [
            CreateAction::make(),
            Action::make('download_template')
                ->label('Download Template')
                ->icon('heroicon-o-document-arrow-down')
                ->color('gray')
                ->url(route('students.template'))
                ->openUrlInNewTab(),
            Action::make('bulk_upload')
                ->label('Bulk Upload')
                ->icon('heroicon-o-arrow-up-tray')
                ->color('success')
                ->form([
                    FileUpload::make('file')
                        ->label('Upload CSV/Excel File')
                        ->acceptedFileTypes(['text/csv', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'])
                        ->maxSize(2048)
                        ->required()
                        ->helperText('Upload a CSV or Excel file with columns: full_name, matriculation_number, session, email, password. Session format: YYYY/YY (e.g., 2024/25). Use the "Download Template" button to get a sample file with proper formatting.')
                ])
                ->action(function (array $data): void {
                    $file = $data['file'];
                    $filePath = storage_path('app/private/' . $file);
                    
                    try {
                        $excelData = Excel::toArray([], $filePath)[0];
                        
                        if (empty($excelData)) {
                            Notification::make()
                                ->title('Error')
                                ->body('The uploaded file is empty.')
                                ->danger()
                                ->send();
                            return;
                        }

                        // Get headers from first row
                        $headers = array_shift($excelData);
                        $headers = array_map('trim', $headers);

                        // Validate required headers
                        $requiredHeaders = ['full_name', 'matriculation_number', 'session'];
                        $missingHeaders = array_diff($requiredHeaders, $headers);
                        
                        if (!empty($missingHeaders)) {
                            Notification::make()
                                ->title('Error')
                                ->body('Missing required columns: ' . implode(', ', $missingHeaders))
                                ->danger()
                                ->send();
                            return;
                        }

                        $successCount = 0;
                        $errorCount = 0;
                        $errors = [];

                        foreach ($excelData as $index => $row) {
                            $rowNumber = $index + 2;
                            
                            // Create associative array from row data
                            $studentData = array_combine($headers, $row);
                            
                            // Validate row data
                            $validator = Validator::make($studentData, [
                                'full_name' => ['required', 'string', 'max:255', 'regex:/^[a-zA-Z\s\.\-\']+$/'],
                                'matriculation_number' => [
                                    'required', 
                                    'string', 
                                    'max:50',
                                    'regex:/^[A-Z0-9\/\-]+$/i',
                                    Rule::unique('students', 'matriculation_number')
                                ],
                                'session' => ['required', 'string', 'max:20', 'regex:/^\d{4}\/\d{2}$/'],
                                'email' => [
                                    'nullable', 
                                    'email', 
                                    'max:255',
                                    Rule::unique('students', 'email')->whereNotNull('email')
                                ],
                                'password' => ['required', 'string', 'min:6', 'max:255'],
                            ]);

                            if ($validator->fails()) {
                                $errorCount++;
                                $errors[] = "Row {$rowNumber}: " . implode(', ', $validator->errors()->all());
                                continue;
                            }

                            try {
                                Student::create([
                                    'full_name' => trim($studentData['full_name']),
                                    'matriculation_number' => trim($studentData['matriculation_number']),
                                    'session' => trim($studentData['session']),
                                    'email' => isset($studentData['email']) && !empty($studentData['email']) ? trim($studentData['email']) : null,
                                    'password' => trim($studentData['password']),
                                    'is_active' => true,
                                ]);
                                $successCount++;
                            } catch (\Exception $e) {
                                $errorCount++;
                                $errors[] = "Row {$rowNumber}: Failed to create student - " . $e->getMessage();
                            }
                        }

                        // Show results
                        if ($successCount > 0) {
                            Notification::make()
                                ->title('Success')
                                ->body("Successfully imported {$successCount} students.")
                                ->success()
                                ->send();
                        }

                        if ($errorCount > 0) {
                            $errorMessage = "Failed to import {$errorCount} students.";
                            if (count($errors) <= 3) {
                                $errorMessage .= "\n\nErrors:\n" . implode("\n", array_slice($errors, 0, 3));
                            } else {
                                $errorMessage .= "\n\nShowing first 3 errors:\n" . implode("\n", array_slice($errors, 0, 3));
                                $errorMessage .= "\n... and " . (count($errors) - 3) . " more errors.";
                            }
                            
                            Notification::make()
                                ->title('Import Warnings')
                                ->body($errorMessage)
                                ->warning()
                                ->send();
                        }

                    } catch (\Exception $e) {
                        Notification::make()
                            ->title('Error')
                            ->body('Failed to process file: ' . $e->getMessage())
                            ->danger()
                            ->send();
                    }
                }),
        ];
    }
}
