<?php

namespace App\Jobs;

use App\Services\QuestionImportService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use App\Notifications\QuestionImportCompleted;

class ProcessQuestionImport implements ShouldQueue
{
    use Queueable, InteractsWithQueue, SerializesModels;

    /**
     * The number of times the job may be attempted.
     */
    public int $tries = 3;

    /**
     * The maximum number of seconds the job can run.
     */
    public int $timeout = 300; // 5 minutes

    /**
     * Create a new job instance.
     */
    public function __construct(
        public array $questions,
        public string $batchId,
        public array $metadata,
        public ?int $userId = null
    ) {
        $this->userId = $userId ?? auth()->id();
    }

    /**
     * Execute the job.
     */
    public function handle(QuestionImportService $importService): void
    {
        try {
            Log::info('Starting question import process', [
                'batch_id' => $this->batchId,
                'question_count' => count($this->questions),
                'user_id' => $this->userId
            ]);

            $imported = $importService->importQuestions($this->questions);

            Log::info('Question import completed successfully', [
                'batch_id' => $this->batchId,
                'imported_count' => $imported,
                'total_questions' => count($this->questions)
            ]);

            // TODO: Add notification when QuestionImportCompleted notification is created

        } catch (\Exception $e) {
            Log::error('Question import failed', [
                'batch_id' => $this->batchId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            throw $e;
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('Question import job failed permanently', [
            'batch_id' => $this->batchId,
            'error' => $exception->getMessage(),
            'attempts' => $this->attempts()
        ]);
    }
}
