<?php

namespace App\Livewire\Auth;

use Livewire\Attributes\Validate;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Validation\ValidationException;

class UnifiedLogin extends Component
{
    #[Validate('required|in:student,staff,applicant', message: 'Please select your user type')]
    public string $user_type = '';

    #[Validate('required|string', message: 'Please enter your login credentials')]
    public string $identifier = '';

    #[Validate('required|string|min:6', message: 'Password must be at least 6 characters')]
    public string $password = '';

    public bool $remember = false;
    public bool $isLoading = false;

    public function mount()
    {
        // Set default user type to student
        $this->user_type = 'student';
    }

    /**
     * Get the identifier field name for the current user type.
     */
    protected function getIdentifierField(): string
    {
        return match ($this->user_type) {
            'student' => 'matriculation_number',
            'staff' => 'staff_number',
            'applicant' => 'application_number',
            default => 'matriculation_number',
        };
    }

    /**
     * Get the guard name for the current user type.
     */
    protected function getGuard(): string
    {
        return match ($this->user_type) {
            'student' => 'student',
            'staff' => 'staff',
            'applicant' => 'applicant',
            default => 'student',
        };
    }

    /**
     * Get the dashboard route for the current user type.
     */
    protected function getDashboardRoute(): string
    {
        return match ($this->user_type) {
            'student' => 'student.dashboard',
            'staff' => 'staff.dashboard',
            'applicant' => 'applicant.dashboard',
            default => 'student.dashboard',
        };
    }

    public function login()
    {
        $this->isLoading = true;

        // Rate limiting
        $key = 'unified-login-attempt:' . request()->ip();
        if (RateLimiter::tooManyAttempts($key, 5)) {
            $seconds = RateLimiter::availableIn($key);
            throw ValidationException::withMessages([
                'identifier' => "Too many login attempts. Please try again in {$seconds} seconds."
            ]);
        }

        $validated = $this->validate();

        // Determine the correct field name based on user type
        $credentials = [
            $this->getIdentifierField() => $this->identifier,
            'password' => $this->password
        ];

        $guard = $this->getGuard();

        if (Auth::guard($guard)->attempt($credentials, $this->remember)) {
            RateLimiter::clear($key);
            session()->regenerate();
            
            // Store the guard type in session for use across the app
            session(['guard' => $guard]);
            
            $route = $this->getDashboardRoute();
            $this->redirect(route($route), navigate: true);
        } else {
            RateLimiter::hit($key);
            $this->addError('identifier', 'Invalid credentials. Please check your details and try again.');
        }

        $this->isLoading = false;
        $this->password = ''; // Clear password for security
    }

    public function render()
    {
        return view('livewire.auth.unified-login');
    }
}
