<?php

namespace App\Livewire\Exam;

use App\Enums\ExamSessionStatus;
use App\Models\ExamSession;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('components.layouts.app')]
class ExamList extends Component
{
    use WithPagination;

    public string $search = '';
    public string $filter = 'available'; // available, upcoming, completed

    protected $queryString = [
        'search' => ['except' => ''],
        'filter' => ['except' => 'available'],
    ];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function getCandidate(): object
    {
        $guard = session('guard', 'student');
        return Auth::guard($guard)->user();
    }

    public function getCandidateType(): string
    {
        $guard = session('guard', 'student');
        return $guard === 'student' ? \App\Models\Student::class : \App\Models\Staff::class;
    }

    public function render()
    {
        $candidate = $this->getCandidate();
        $candidateType = $this->getCandidateType();
        $guard = session('guard', 'student');
        $audience = $guard === 'student' ? 'student' : 'staff';
        $examSettings = app(\App\Settings\ExamSettings::class);

        $query = ExamSession::with(['exam'])
            ->whereHas('exam', function ($q) use ($audience) {
                $q->where('is_active', true)
                  ->where(function ($q2) use ($audience) {
                      $q2->where('target_audience', $audience)
                         ->orWhere('target_audience', 'both');
                  });
            })
            ->accessibleBy($candidateType, $candidate->id);

        // Apply search
        if ($this->search) {
            $query->whereHas('exam', function ($q) {
                $q->where('title', 'like', "%{$this->search}%")
                  ->orWhere('code', 'like', "%{$this->search}%");
            });
        }

        // Apply filters
        switch ($this->filter) {
            case 'available':
                $query->where('status', ExamSessionStatus::ACTIVE)
                      ->where('start_time', '<=', now())
                      ->where('end_time', '>=', now());
                break;
            case 'upcoming':
                $query->whereIn('status', [ExamSessionStatus::SCHEDULED, ExamSessionStatus::ACTIVE])
                      ->where('start_time', '>', now());
                break;
            case 'completed':
                $query->whereHas('attempts', function ($q) use ($candidateType, $candidate) {
                    $q->where('candidate_type', $candidateType)
                      ->where('candidate_id', $candidate->id)
                      ->where('status', 'completed');
                });
                break;
        }

        $sessions = $query->orderBy('start_time')->paginate(10);

        // Add extra info for each session
        foreach ($sessions as $session) {
            $session->can_take = $session->canCandidateTake($candidateType, $candidate->id);
            $session->my_attempts = $session->attempts()
                ->where('candidate_type', $candidateType)
                ->where('candidate_id', $candidate->id)
                ->get();
        }

        return view('livewire.exam.exam-list', [
            'sessions' => $sessions,
            'candidate' => $candidate,
            'showResults' => $examSettings->show_results_to_students,
        ]);
    }
}
