<?php

namespace App\Livewire\Exam;

use App\Models\ExamAttempt;
use App\Settings\ExamSettings;
use App\Settings\TerminologySettings;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class ExamResult extends Component
{
    public ExamAttempt $attempt;
    public bool $showDetailedResults = true;
    public bool $showCorrectAnswers = true;
    public bool $showResults = true;

    public function mount(ExamAttempt $attempt)
    {
        $this->attempt = $attempt->load(['examSession.exam', 'answers.question']);

        // Verify ownership
        $guard = session('guard', 'student');
        $candidate = Auth::guard($guard)->user();
        $candidateType = $guard === 'student' ? \App\Models\Student::class : \App\Models\Staff::class;

        if ($attempt->candidate_type !== $candidateType || $attempt->candidate_id !== $candidate->id) {
            abort(403, 'Unauthorized');
        }

        // Get global exam settings
        $examSettings = app(ExamSettings::class);
        
        // Settings logic:
        // - show_results_to_students: PRIMARY gate - if false, students cannot access results page at all
        // - show_score_immediately: If true, show score on the redirect message (but still redirect away)
        // - show_correct_answers_to_students: Show which answers were correct/incorrect in the detail view
        
        // If results are not enabled for students, redirect to exam list
        if (!$examSettings->show_results_to_students) {
            $message = $examSettings->show_score_immediately
                ? "Your exam has been submitted. Score: {$attempt->score_percentage}%. Results details will be available once released by the administrator."
                : 'Your exam has been submitted. Results will be available once released by the administrator.';
            
            session()->flash('info', $message);
            return $this->redirect(route('exam.list'), navigate: true);
        }
        
        // Determine what to show on the results page
        $this->showResults = true; // Already gated above
        $this->showCorrectAnswers = $examSettings->show_correct_answers_to_students 
            && ($this->attempt->examSession->exam->show_correct_answers ?? true);
        
        // Show detailed question review only if correct answers visibility is enabled
        // Don't show detailed results if there are pending essays (no correct answer to show)
        $this->showDetailedResults = $this->showCorrectAnswers && !$this->attempt->is_all_essay;
    }

    /**
     * Check if there are essays pending grading.
     */
    public function getHasPendingEssaysProperty(): bool
    {
        return $this->attempt->has_pending_essays;
    }

    /**
     * Check if exam is all essay questions.
     */
    public function getIsAllEssayProperty(): bool
    {
        return $this->attempt->is_all_essay;
    }

    public function backToExams()
    {
        return $this->redirect(route('exam.list'), navigate: true);
    }

    public function getStatisticsProperty(): array
    {
        return [
            'correct' => $this->attempt->correct_answers,
            'incorrect' => $this->attempt->wrong_answers,
            'skipped' => $this->attempt->skipped_questions,
            'total' => $this->attempt->total_questions,
        ];
    }

    public function getAnswersProperty(): array
    {
        $answersData = [];
        
        foreach ($this->attempt->answers()->with('question')->orderBy('question_index')->get() as $answer) {
            $question = $answer->question;
            $answersData[] = [
                'question_text' => $question->question_text ?? 'Question not found',
                'options' => $question->formatted_options ?? [],
                'selected_answer' => $answer->selected_answer,
                'correct_answer' => $answer->correct_answer,
                'is_correct' => $answer->is_correct,
                'is_skipped' => $answer->is_skipped,
                'marks_obtained' => $answer->marks_obtained,
                'marks_possible' => $answer->marks_possible,
            ];
        }

        return $answersData;
    }

    public function render()
    {
        $guard = session('guard', 'student');
        $user = Auth::guard($guard)->user();
        $terminology = app(TerminologySettings::class);
        
        return view('livewire.exam.exam-result', [
            'exam' => $this->attempt->examSession->exam,
            'session' => $this->attempt->examSession,
            'statistics' => $this->statistics,
            'answers' => $this->answers,
            'showResults' => $this->showResults,
            'showCorrectAnswers' => $this->showCorrectAnswers,
        ])->layout('layouts.exam', [
            'title' => 'Exam Results',
            'userName' => $user->name ?? $user->full_name ?? 'Candidate',
            'userRole' => $guard === 'student' ? $terminology->primary_user_singular : $terminology->secondary_user_singular,
            'showTimer' => false,
            'timeRemaining' => 0,
            'timerDisplay' => '',
        ]);
    }
}
