<?php

namespace App\Models;

use App\Enums\QuestionSelectionType;
use App\Settings\TerminologySettings;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class Exam extends Model
{
    use HasFactory, SoftDeletes, LogsActivity;

    public const TARGET_AUDIENCES = [
        'student',
        'staff',
        'applicant',
        'both',
        'all',
    ];

    public const QUESTION_TYPES = [
        'mcq' => 'Multiple Choice Questions',
        'essay' => 'Essay Questions',
    ];

    protected $fillable = [
        'title',
        'code',
        'description',
        'instructions',
        'duration_minutes',
        'total_questions',
        'passing_score',
        'marks_per_question',
        'negative_marking',
        'question_selection_type',
        'question_ids',
        'category_distribution',
        'target_audience',
        'question_type',
        'target_period',
        'target_department',
        'shuffle_questions',
        'shuffle_options',
        'show_result_immediately',
        'allow_review',
        'show_correct_answers',
        'is_active',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'question_ids' => 'array',
        'category_distribution' => 'array',
        'shuffle_questions' => 'boolean',
        'shuffle_options' => 'boolean',
        'show_result_immediately' => 'boolean',
        'allow_review' => 'boolean',
        'show_correct_answers' => 'boolean',
        'is_active' => 'boolean',
        'duration_minutes' => 'integer',
        'total_questions' => 'integer',
        'passing_score' => 'decimal:2',
        'marks_per_question' => 'decimal:2',
        'negative_marking' => 'decimal:2',
        'question_selection_type' => QuestionSelectionType::class,
    ];

    protected $attributes = [
        'duration_minutes' => 60,
        'passing_score' => 50.00,
        'marks_per_question' => 1,
        'negative_marking' => 0,
        'question_selection_type' => 'manual',
        'shuffle_questions' => true,
        'shuffle_options' => true,
        'show_result_immediately' => true,
        'allow_review' => true,
        'show_correct_answers' => false,
        'is_active' => true,
    ];

    /**
     * Boot the model.
     */
    protected static function boot(): void
    {
        parent::boot();

        static::creating(function (self $exam) {
            if (empty($exam->code)) {
                $exam->code = self::generateUniqueCode();
            }
        });
    }

    /**
     * Generate a unique exam code.
     */
    public static function generateUniqueCode(): string
    {
        do {
            $code = 'EX-' . strtoupper(Str::random(6));
        } while (self::where('code', $code)->exists());

        return $code;
    }

    /**
     * Activity Log Configuration
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['title', 'code', 'duration_minutes', 'total_questions', 'passing_score', 'is_active'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /**
     * Check if this is an essay exam.
     */
    public function isEssay(): bool
    {
        return $this->question_type === 'essay';
    }

    /**
     * Check if this is an MCQ exam.
     */
    public function isMcq(): bool
    {
        return $this->question_type === 'mcq' || $this->question_type === null;
    }

    // Relationships
    public function sessions(): HasMany
    {
        return $this->hasMany(ExamSession::class);
    }

    public function activeSessions(): HasMany
    {
        return $this->hasMany(ExamSession::class)->where('status', 'active');
    }

    public function questions(): BelongsToMany
    {
        return $this->belongsToMany(QuestionBank::class, 'exam_questions')
            ->withPivot('sort_order')
            ->orderByPivot('sort_order');
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Get staff assigned as examiners for this exam.
     */
    public function examiners(): BelongsToMany
    {
        return $this->belongsToMany(Staff::class, 'exam_examiners')
            ->withPivot(['role', 'can_grade', 'can_edit_questions'])
            ->withTimestamps();
    }

    /**
     * Get the lead examiner for this exam.
     */
    public function leadExaminer(): BelongsToMany
    {
        return $this->examiners()->wherePivot('role', 'lead_examiner');
    }

    /**
     * Check if a staff member is an examiner for this exam.
     */
    public function isExaminer(Staff $staff): bool
    {
        return $this->examiners()->where('staff_id', $staff->id)->exists();
    }

    /**
     * Check if a staff member can grade this exam.
     */
    public function canGrade(Staff $staff): bool
    {
        return $this->examiners()
            ->where('staff_id', $staff->id)
            ->wherePivot('can_grade', true)
            ->exists();
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeForAudience($query, string $audience)
    {
        return $query->where(function ($q) use ($audience) {
            $q->where('target_audience', $audience)
              ->orWhere('target_audience', 'both');
        });
    }

    // Helpers
    /**
     * Get target audience options with dynamic terminology
     */
    public static function getTargetAudienceOptions(): array
    {
        $terminology = app(TerminologySettings::class);
        
        return [
            'student' => $terminology->primary_user_plural,
            'staff' => $terminology->secondary_user_plural,
            'applicant' => 'Applicants',
            'both' => $terminology->primary_user_plural . ' & ' . $terminology->secondary_user_plural,
            'all' => 'All Users',
        ];
    }

    /**
     * Get formatted target audience label
     */
    public function getTargetAudienceLabelAttribute(): string
    {
        $options = self::getTargetAudienceOptions();
        return $options[$this->target_audience] ?? $this->target_audience;
    }

    /**
     * Get the total marks for this exam.
     */
    public function getTotalMarksAttribute(): int
    {
        return $this->total_questions * $this->marks_per_question;
    }

    /**
     * Get questions for an exam attempt based on selection type.
     */
    public function getQuestionsForAttempt(): array
    {
        $questions = collect();
        $isEssay = $this->isEssay();

        switch ($this->question_selection_type) {
            case QuestionSelectionType::MANUAL:
                if ($this->question_ids) {
                    $query = QuestionBank::whereIn('id', $this->question_ids)
                        ->where('is_active', true);
                    
                    // Filter by question type - essay vs MCQ types
                    if ($isEssay) {
                        $query->where('question_type', 'essay');
                    } else {
                        $query->whereIn('question_type', ['single_select', 'true_false', 'multiple_select']);
                    }
                    
                    $questions = $query->get();
                }
                break;

            case QuestionSelectionType::RANDOM:
                $query = QuestionBank::where('is_active', true)
                    ->where('target_audience', $this->target_audience)
                    ->when($this->target_period, fn($q) => $q->where('session', $this->target_period))
                    ->when($this->target_department, fn($q) => $q->where('department', $this->target_department));
                
                // Filter by question type
                if ($isEssay) {
                    $query->where('question_type', 'essay');
                } else {
                    $query->whereIn('question_type', ['single_select', 'true_false', 'multiple_select']);
                }
                
                $questions = $query->inRandomOrder()
                    ->limit($this->total_questions)
                    ->get();
                break;

            case QuestionSelectionType::CATEGORY_BASED:
                if ($this->category_distribution) {
                    // category_distribution is now an array of category IDs
                    // Get ALL questions from the selected categories of the correct type AND target audience
                    $categoryIds = is_array($this->category_distribution) 
                        ? $this->category_distribution 
                        : [];
                    
                    $query = QuestionBank::whereIn('category_id', $categoryIds)
                        ->where('is_active', true);
                    
                    // Filter by question type
                    if ($isEssay) {
                        $query->where('question_type', 'essay');
                    } else {
                        $query->whereIn('question_type', ['single_select', 'true_false', 'multiple_select']);
                    }
                    
                    // Filter by target audience to ensure only matching questions are included
                    if (in_array($this->target_audience, ['student', 'staff', 'applicant'])) {
                        $query->where('target_audience', $this->target_audience);
                    } elseif ($this->target_audience === 'both') {
                        // 'both' means students and staff
                        $query->whereIn('target_audience', ['student', 'staff']);
                    }
                    // 'all' means no audience filter - include all questions
                    
                    $questions = $query->get();
                }
                break;
        }

        // Shuffle if enabled
        if ($this->shuffle_questions) {
            $questions = $questions->shuffle();
        }

        return $questions->pluck('id')->toArray();
    }

    /**
     * Get the passing marks threshold.
     */
    public function getPassingMarksAttribute(): float
    {
        return ($this->passing_score / 100) * $this->total_marks;
    }

    /**
     * Check if a candidate has already attempted this exam in a session.
     */
    public function hasAttempted(string $candidateType, int $candidateId, int $sessionId): bool
    {
        return ExamAttempt::where('exam_session_id', $sessionId)
            ->where('candidate_type', $candidateType)
            ->where('candidate_id', $candidateId)
            ->exists();
    }

    /**
     * Get attempt count for a candidate in a session.
     */
    public function getAttemptCount(string $candidateType, int $candidateId, int $sessionId): int
    {
        return ExamAttempt::where('exam_session_id', $sessionId)
            ->where('candidate_type', $candidateType)
            ->where('candidate_id', $candidateId)
            ->count();
    }
}
