<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ExamAnswer extends Model
{
    use HasFactory;

    protected $fillable = [
        'exam_attempt_id',
        'question_bank_id',
        'question_index',
        'selected_answer',
        'essay_answer',
        'correct_answer',
        'is_correct',
        'marks_obtained',
        'marks_possible',
        'time_spent_seconds',
        'answered_at',
        'is_flagged',
        'is_skipped',
        'requires_grading',
        'is_graded',
        'graded_by',
        'graded_at',
        'grading_feedback',
    ];

    protected $casts = [
        'is_correct' => 'boolean',
        'is_flagged' => 'boolean',
        'is_skipped' => 'boolean',
        'requires_grading' => 'boolean',
        'is_graded' => 'boolean',
        'marks_obtained' => 'decimal:2',
        'marks_possible' => 'integer',
        'time_spent_seconds' => 'integer',
        'question_index' => 'integer',
        'answered_at' => 'datetime',
        'graded_at' => 'datetime',
    ];

    protected $attributes = [
        'is_flagged' => false,
        'is_skipped' => true,
        'marks_obtained' => 0,
        'marks_possible' => 1,
        'time_spent_seconds' => 0,
    ];

    // Relationships
    public function attempt(): BelongsTo
    {
        return $this->belongsTo(ExamAttempt::class, 'exam_attempt_id');
    }

    public function question(): BelongsTo
    {
        return $this->belongsTo(QuestionBank::class, 'question_bank_id');
    }

    public function grader(): BelongsTo
    {
        return $this->belongsTo(Staff::class, 'graded_by');
    }

    // Scopes
    public function scopeAnswered($query)
    {
        return $query->whereNotNull('selected_answer');
    }

    public function scopeEssay($query)
    {
        return $query->whereNotNull('essay_answer');
    }

    public function scopeRequiresGrading($query)
    {
        return $query->where('requires_grading', true)->where('is_graded', false);
    }

    public function scopeGraded($query)
    {
        return $query->where('is_graded', true);
    }

    public function scopeCorrect($query)
    {
        return $query->where('is_correct', true);
    }

    public function scopeIncorrect($query)
    {
        return $query->where('is_correct', false)->whereNotNull('selected_answer');
    }

    public function scopeSkipped($query)
    {
        return $query->whereNull('selected_answer');
    }

    public function scopeFlagged($query)
    {
        return $query->where('is_flagged', true);
    }

    // Helpers
    /**
     * Check if this answer is correct.
     */
    public function isCorrect(): bool
    {
        return $this->is_correct === true;
    }

    /**
     * Check if this answer was skipped.
     */
    public function isSkipped(): bool
    {
        return $this->selected_answer === null;
    }

    /**
     * Get the selected option text from the question.
     */
    public function getSelectedOptionTextAttribute(): ?string
    {
        if ($this->selected_answer === null) {
            return null;
        }

        $question = $this->question;
        if (!$question || !$question->options) {
            return $this->selected_answer;
        }

        return $question->options[$this->selected_answer] ?? $this->selected_answer;
    }

    /**
     * Get the correct option text from the question.
     */
    public function getCorrectOptionTextAttribute(): ?string
    {
        $question = $this->question;
        if (!$question || !$question->options) {
            return $this->correct_answer;
        }

        return $question->options[$this->correct_answer] ?? $this->correct_answer;
    }

    /**
     * Get formatted time spent.
     */
    public function getFormattedTimeSpentAttribute(): string
    {
        $seconds = $this->time_spent_seconds;
        if ($seconds < 60) {
            return "{$seconds} sec";
        }

        $minutes = floor($seconds / 60);
        $secs = $seconds % 60;
        return "{$minutes}m {$secs}s";
    }

    /**
     * Get status label.
     */
    public function getStatusLabelAttribute(): string
    {
        if ($this->isSkipped()) {
            return 'Skipped';
        }

        return $this->isCorrect() ? 'Correct' : 'Incorrect';
    }

    /**
     * Get status color for display.
     */
    public function getStatusColorAttribute(): string
    {
        if ($this->isSkipped()) {
            return 'gray';
        }

        return $this->isCorrect() ? 'success' : 'danger';
    }
}
