<?php

namespace App\Models;

use App\Enums\ExamAttemptStatus;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class ExamAttempt extends Model
{
    use HasFactory, LogsActivity;

    protected $fillable = [
        'exam_session_id',
        'candidate_type',
        'candidate_id',
        'question_order',
        'started_at',
        'completed_at',
        'time_spent_seconds',
        'total_questions',
        'answered_questions',
        'correct_answers',
        'wrong_answers',
        'skipped_questions',
        'score_obtained',
        'score_percentage',
        'passed',
        'status',
        'current_question_index',
        'ip_address',
        'user_agent',
    ];

    protected $casts = [
        'question_order' => 'array',
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
        'time_spent_seconds' => 'integer',
        'total_questions' => 'integer',
        'answered_questions' => 'integer',
        'correct_answers' => 'integer',
        'wrong_answers' => 'integer',
        'skipped_questions' => 'integer',
        'score_obtained' => 'decimal:2',
        'score_percentage' => 'decimal:2',
        'passed' => 'boolean',
        'current_question_index' => 'integer',
        'status' => ExamAttemptStatus::class,
    ];

    protected $attributes = [
        'time_spent_seconds' => 0,
        'total_questions' => 0,
        'answered_questions' => 0,
        'correct_answers' => 0,
        'wrong_answers' => 0,
        'skipped_questions' => 0,
        'score_obtained' => 0,
        'score_percentage' => 0,
        'current_question_index' => 0,
        'status' => 'in_progress',
    ];

    /**
     * Activity Log Configuration
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['status', 'score_percentage', 'passed', 'completed_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    // Relationships
    public function examSession(): BelongsTo
    {
        return $this->belongsTo(ExamSession::class);
    }

    public function candidate(): MorphTo
    {
        return $this->morphTo('candidate', 'candidate_type', 'candidate_id');
    }

    public function answers(): HasMany
    {
        return $this->hasMany(ExamAnswer::class);
    }

    // Scopes
    public function scopeInProgress($query)
    {
        return $query->where('status', ExamAttemptStatus::IN_PROGRESS);
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', ExamAttemptStatus::COMPLETED);
    }

    public function scopeForCandidate($query, string $type, int $id)
    {
        return $query->where('candidate_type', $type)->where('candidate_id', $id);
    }

    // Helpers
    /**
     * Get the exam for this attempt.
     */
    public function getExamAttribute(): ?Exam
    {
        return $this->examSession?->exam;
    }

    /**
     * Check if the attempt is still in progress.
     */
    public function isInProgress(): bool
    {
        return $this->status === ExamAttemptStatus::IN_PROGRESS;
    }

    /**
     * Check if the attempt has been completed.
     */
    public function isCompleted(): bool
    {
        return $this->status === ExamAttemptStatus::COMPLETED;
    }

    /**
     * Get time remaining in seconds.
     */
    public function getTimeRemainingAttribute(): int
    {
        if (!$this->isInProgress()) {
            return 0;
        }

        $exam = $this->exam;
        if (!$exam) {
            return 0;
        }

        $durationSeconds = $exam->duration_minutes * 60;
        $elapsed = now()->diffInSeconds($this->started_at);
        
        return max(0, $durationSeconds - $elapsed);
    }

    /**
     * Get formatted time remaining.
     */
    public function getFormattedTimeRemainingAttribute(): string
    {
        $seconds = $this->time_remaining;
        $minutes = floor($seconds / 60);
        $secs = $seconds % 60;
        
        return sprintf('%02d:%02d', $minutes, $secs);
    }

    /**
     * Check if the attempt has timed out.
     */
    public function hasTimedOut(): bool
    {
        return $this->time_remaining <= 0 && $this->isInProgress();
    }

    /**
     * Get the current question.
     */
    public function getCurrentQuestion(): ?QuestionBank
    {
        if (empty($this->question_order) || $this->current_question_index >= count($this->question_order)) {
            return null;
        }

        $questionId = $this->question_order[$this->current_question_index];
        return QuestionBank::find($questionId);
    }

    /**
     * Get a question by index.
     */
    public function getQuestionAtIndex(int $index): ?QuestionBank
    {
        if (empty($this->question_order) || $index < 0 || $index >= count($this->question_order)) {
            return null;
        }

        $questionId = $this->question_order[$index];
        return QuestionBank::find($questionId);
    }

    /**
     * Get the answer for a specific question.
     */
    public function getAnswerForQuestion(int $questionId): ?ExamAnswer
    {
        return $this->answers()->where('question_bank_id', $questionId)->first();
    }

    /**
     * Submit an answer for a question.
     */
    public function submitAnswer(int $questionId, ?string $selectedAnswer, int $timeSpent = 0, ?string $essayAnswer = null): ExamAnswer
    {
        $question = QuestionBank::findOrFail($questionId);
        
        // Handle essay questions differently
        if ($question->isEssay()) {
            return $this->submitEssayAnswer($question, $essayAnswer, $timeSpent);
        }
        
        $isCorrect = $selectedAnswer !== null && strcasecmp($selectedAnswer, $question->correct_answer) === 0;
        
        $answer = $this->answers()->updateOrCreate(
            ['question_bank_id' => $questionId],
            [
                'question_index' => array_search($questionId, $this->question_order) + 1,
                'selected_answer' => $selectedAnswer,
                'correct_answer' => $question->correct_answer,
                'is_correct' => $selectedAnswer !== null ? $isCorrect : null,
                'marks_obtained' => (float) ($isCorrect ? ($this->exam->marks_per_question ?? 1) : ($selectedAnswer !== null ? -($this->exam->negative_marking ?? 0) : 0)),
                'marks_possible' => (int) ($this->exam->marks_per_question ?? 1),
                'time_spent_seconds' => $timeSpent,
                'answered_at' => $selectedAnswer !== null ? now() : null,
                'is_skipped' => $selectedAnswer === null,
            ]
        );

        // Update attempt stats
        $this->updateStats();

        return $answer;
    }

    /**
     * Submit an essay answer for a question.
     */
    protected function submitEssayAnswer(QuestionBank $question, ?string $essayAnswer, int $timeSpent = 0): ExamAnswer
    {
        $hasAnswer = $essayAnswer !== null && trim($essayAnswer) !== '';
        
        // Ensure we have the exam loaded to get marks_per_question
        $this->loadMissing('examSession.exam');
        $marksPerQuestion = $this->examSession?->exam?->marks_per_question ?? 1;
        
        $answer = $this->answers()->updateOrCreate(
            ['question_bank_id' => $question->id],
            [
                'question_index' => array_search($question->id, $this->question_order) + 1,
                'essay_answer' => $essayAnswer,
                'selected_answer' => null, // Essay questions don't have selected answers
                'correct_answer' => null, // Essay questions don't have a single correct answer
                'is_correct' => null, // Requires manual grading
                'marks_obtained' => 0, // Will be assigned during grading
                'marks_possible' => (int) $marksPerQuestion,
                'time_spent_seconds' => $timeSpent,
                'answered_at' => $hasAnswer ? now() : null,
                'is_skipped' => !$hasAnswer,
                'requires_grading' => $hasAnswer, // Mark for manual grading if answered
                'is_graded' => false,
            ]
        );

        // Update attempt stats
        $this->updateStats();

        return $answer;
    }

    /**
     * Update statistics for this attempt.
     */
    public function updateStats(): void
    {
        $answers = $this->answers;
        
        // Count answered questions (both MCQ with selected_answer and essays with essay_answer)
        $answered = $answers->filter(function ($answer) {
            // Has MCQ answer
            if ($answer->selected_answer !== null) {
                return true;
            }
            // Has essay answer
            if ($answer->essay_answer !== null && trim($answer->essay_answer) !== '') {
                return true;
            }
            return false;
        })->count();
        
        $correct = $answers->where('is_correct', true)->count();
        $wrong = $answers->where('is_correct', false)->whereNotNull('selected_answer')->count();
        $skipped = $this->total_questions - $answered;
        
        $scoreObtained = $answers->sum('marks_obtained');
        $totalMarks = (float) ($this->total_questions * ($this->exam->marks_per_question ?? 1));
        $scorePercentage = $totalMarks > 0 ? (float) (($scoreObtained / $totalMarks) * 100) : 0;
        
        $endTime = $this->completed_at ?? now();
        $timeSpent = $this->started_at ? (int) $this->started_at->diffInSeconds($endTime) : 0;
        
        // Recalculate passed status based on updated score
        $passingScore = $this->exam->passing_score ?? 50;
        $passed = $scorePercentage >= $passingScore;
        
        $this->update([
            'answered_questions' => $answered,
            'correct_answers' => $correct,
            'wrong_answers' => $wrong,
            'skipped_questions' => $skipped,
            'score_obtained' => max(0, $scoreObtained),
            'score_percentage' => max(0, $scorePercentage),
            'time_spent_seconds' => max(0, $timeSpent),
            'passed' => $passed,
        ]);
    }

    /**
     * Complete the exam attempt.
     */
    public function complete(string $status = 'completed'): void
    {
        // Set completed_at first so updateStats can calculate time correctly
        $this->update([
            'completed_at' => now(),
        ]);
        
        // Refresh model to get updated completed_at
        $this->refresh();
        
        $this->updateStats();
        
        $this->update([
            'status' => $status,
            'passed' => $this->score_percentage >= ($this->exam->passing_score ?? 50),
        ]);
    }

    /**
     * Flag a question for review.
     */
    public function flagQuestion(int $questionId, bool $flagged = true): void
    {
        $this->answers()
            ->where('question_bank_id', $questionId)
            ->update(['is_flagged' => $flagged]);
    }

    /**
     * Get all flagged questions.
     */
    public function getFlaggedQuestions(): array
    {
        return $this->answers()
            ->where('is_flagged', true)
            ->pluck('question_bank_id')
            ->toArray();
    }

    /**
     * Get progress summary.
     */
    public function getProgressSummary(): array
    {
        $total = count($this->question_order ?? []);
        $answered = $this->answered_questions;
        $flagged = $this->answers()->where('is_flagged', true)->count();

        return [
            'total' => $total,
            'answered' => $answered,
            'skipped' => $total - $answered,
            'flagged' => $flagged,
            'current_index' => $this->current_question_index,
            'progress_percentage' => $total > 0 ? round(($answered / $total) * 100) : 0,
        ];
    }

    /**
     * Get navigation data for all questions.
     */
    public function getNavigationData(): array
    {
        $data = [];
        $answersMap = $this->answers->keyBy('question_bank_id');
        $questionsMap = QuestionBank::whereIn('id', $this->question_order)->get()->keyBy('id');

        foreach ($this->question_order as $index => $questionId) {
            $answer = $answersMap->get($questionId);
            $question = $questionsMap->get($questionId);
            
            // Check if answered based on question type
            $isAnswered = false;
            if ($answer) {
                if ($question && $question->isEssay()) {
                    // For essay questions, check essay_answer
                    $isAnswered = $answer->essay_answer !== null && trim($answer->essay_answer) !== '';
                } else {
                    // For MCQ/other questions, check selected_answer
                    $isAnswered = $answer->selected_answer !== null;
                }
            }
            
            $data[] = [
                'index' => $index,
                'question_id' => $questionId,
                'is_answered' => $isAnswered,
                'is_flagged' => $answer && $answer->is_flagged,
                'is_current' => $index === $this->current_question_index,
            ];
        }

        return $data;
    }

    /**
     * Get candidate name.
     */
    public function getCandidateNameAttribute(): string
    {
        $candidate = $this->candidate;
        if (!$candidate) {
            return 'Unknown';
        }

        // Handle different candidate types
        if ($candidate instanceof Student) {
            return $candidate->name ?? $candidate->full_name ?? 'Unknown Student';
        }

        if ($candidate instanceof Staff) {
            return $candidate->name ?? $candidate->full_name ?? 'Unknown Staff';
        }

        return $candidate->name ?? 'Unknown';
    }

    /**
     * Get candidate identifier (ID number).
     */
    public function getCandidateIdentifierAttribute(): string
    {
        $candidate = $this->candidate;
        if (!$candidate) {
            return 'N/A';
        }

        if ($candidate instanceof Student) {
            return $candidate->matriculation_number ?? 'N/A';
        }

        if ($candidate instanceof Staff) {
            return $candidate->staff_number ?? 'N/A';
        }

        return $candidate->id ?? 'N/A';
    }

    /**
     * Check if the exam was passed.
     */
    public function getIsPassedAttribute(): bool
    {
        return $this->passed ?? false;
    }

    /**
     * Check if there are essays pending grading.
     */
    public function getHasPendingEssaysAttribute(): bool
    {
        return $this->answers()->where('requires_grading', true)->where('is_graded', false)->exists();
    }

    /**
     * Count essays pending grading.
     */
    public function getPendingEssayCountAttribute(): int
    {
        return $this->answers()->where('requires_grading', true)->where('is_graded', false)->count();
    }

    /**
     * Check if all questions are essay type (for result display logic).
     */
    public function getIsAllEssayAttribute(): bool
    {
        $essayCount = $this->answers()
            ->whereHas('question', fn($q) => $q->where('question_type', 'essay'))
            ->count();
        
        return $essayCount === $this->total_questions && $this->total_questions > 0;
    }

    /**
     * Get total score (alias for score_obtained).
     */
    public function getTotalScoreAttribute(): float
    {
        return $this->score_obtained ?? 0;
    }

    /**
     * Get max possible score.
     */
    public function getMaxScoreAttribute(): int
    {
        return $this->total_questions * ($this->exam?->marks_per_question ?? 1);
    }
}
