<?php

namespace App\Models;

use App\Enums\ExamAttemptStatus;
use App\Enums\ExamSessionStatus;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class ExamSession extends Model
{
    use HasFactory, LogsActivity;

    protected $fillable = [
        'exam_id',
        'title',
        'start_time',
        'end_time',
        'max_attempts',
        'max_participants',
        'access_code',
        'allowed_candidates',
        'status',
        'notes',
        'created_by',
    ];

    protected $casts = [
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'allowed_candidates' => 'array',
        'max_attempts' => 'integer',
        'max_participants' => 'integer',
        'status' => ExamSessionStatus::class,
    ];

    protected $attributes = [
        'max_attempts' => 1,
        'status' => 'scheduled',
    ];

    /**
     * Activity Log Configuration
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['title', 'start_time', 'end_time', 'status', 'max_attempts'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    // Relationships
    public function exam(): BelongsTo
    {
        return $this->belongsTo(Exam::class);
    }

    public function attempts(): HasMany
    {
        return $this->hasMany(ExamAttempt::class);
    }

    public function completedAttempts(): HasMany
    {
        return $this->hasMany(ExamAttempt::class)->where('status', 'completed');
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', ExamSessionStatus::ACTIVE);
    }

    public function scopeScheduled($query)
    {
        return $query->where('status', ExamSessionStatus::SCHEDULED);
    }

    public function scopeUpcoming($query)
    {
        return $query->where('start_time', '>', now());
    }

    public function scopeOngoing($query)
    {
        return $query->where('start_time', '<=', now())
                     ->where('end_time', '>=', now());
    }

    public function scopeAccessibleBy($query, string $candidateType, int $candidateId)
    {
        return $query->where(function ($q) use ($candidateType, $candidateId) {
            $q->whereNull('allowed_candidates')
              ->orWhereJsonContains('allowed_candidates', [
                  'type' => $candidateType,
                  'id' => $candidateId,
              ]);
        });
    }

    // Helpers
    /**
     * Check if the session is currently active (within time window).
     */
    public function isWithinTimeWindow(): bool
    {
        $now = now();
        return $now->gte($this->start_time) && $now->lte($this->end_time);
    }

    /**
     * Check if the session has started.
     */
    public function hasStarted(): bool
    {
        return now()->gte($this->start_time);
    }

    /**
     * Check if the session has ended.
     */
    public function hasEnded(): bool
    {
        return now()->gt($this->end_time);
    }

    /**
     * Get the time remaining until session starts.
     */
    public function getTimeUntilStartAttribute(): ?string
    {
        if ($this->hasStarted()) {
            return null;
        }
        
        return $this->start_time->diffForHumans();
    }

    /**
     * Get the time remaining until session ends.
     */
    public function getTimeRemainingAttribute(): ?string
    {
        if ($this->hasEnded()) {
            return null;
        }

        return $this->end_time->diffForHumans();
    }

    /**
     * Get the duration of the session in hours.
     */
    public function getSessionDurationAttribute(): string
    {
        return $this->start_time->diff($this->end_time)->format('%H:%I');
    }

    /**
     * Check if a candidate can take this exam session.
     */
    public function canCandidateTake(string $candidateType, int $candidateId): array
    {
        $reasons = [];

        // Check for in-progress attempt first - resuming is always allowed if session is active
        $inProgress = $this->attempts()
            ->where('candidate_type', $candidateType)
            ->where('candidate_id', $candidateId)
            ->where('status', ExamAttemptStatus::IN_PROGRESS)
            ->first();

        // Check session status
        if ($this->status !== ExamSessionStatus::ACTIVE) {
            $reasons[] = 'This exam session is not currently active.';
        }

        // Check time window
        if (!$this->isWithinTimeWindow()) {
            if (!$this->hasStarted()) {
                $reasons[] = "This session hasn't started yet. It will begin {$this->time_until_start}.";
            } else {
                $reasons[] = 'This exam session has ended.';
            }
        }

        // If there's an in-progress attempt and session is active/within time window, allow resuming
        if ($inProgress && empty($reasons)) {
            return [
                'can_take' => true,
                'reasons' => [],
                'in_progress_attempt' => $inProgress,
                'attempts_remaining' => 1, // Current attempt
            ];
        }

        // Check allowed candidates if restricted
        if ($this->allowed_candidates) {
            $allowed = collect($this->allowed_candidates)->contains(function ($candidate) use ($candidateType, $candidateId) {
                return $candidate['type'] === $candidateType && $candidate['id'] === $candidateId;
            });

            if (!$allowed) {
                $reasons[] = 'You are not authorized to take this exam session.';
            }
        }

        // Check max participants
        if ($this->max_participants) {
            $currentParticipants = $this->attempts()->distinct('candidate_id', 'candidate_type')->count();
            if ($currentParticipants >= $this->max_participants) {
                $reasons[] = 'This session has reached maximum participants.';
            }
        }

        // Check max attempts - only count completed/submitted attempts, not in-progress
        $completedAttemptCount = $this->attempts()
            ->where('candidate_type', $candidateType)
            ->where('candidate_id', $candidateId)
            ->whereIn('status', [
                ExamAttemptStatus::COMPLETED,
                ExamAttemptStatus::TIMED_OUT,
                ExamAttemptStatus::ABANDONED,
            ])
            ->count();

        if ($completedAttemptCount >= $this->max_attempts) {
            $reasons[] = "You have reached the maximum number of attempts ({$this->max_attempts}) for this session.";
        }

        return [
            'can_take' => empty($reasons),
            'reasons' => $reasons,
            'in_progress_attempt' => $inProgress,
            'attempts_remaining' => max(0, $this->max_attempts - $completedAttemptCount),
        ];
    }

    /**
     * Auto-update status based on time.
     */
    public function updateStatusBasedOnTime(): void
    {
        $now = now();

        if ($this->status === ExamSessionStatus::SCHEDULED && $now->gte($this->start_time)) {
            $this->update(['status' => ExamSessionStatus::ACTIVE]);
        }

        if ($this->status === ExamSessionStatus::ACTIVE && $now->gt($this->end_time)) {
            $this->update(['status' => ExamSessionStatus::COMPLETED]);
        }
    }

    /**
     * Get full title including exam name.
     */
    public function getFullTitleAttribute(): string
    {
        $title = $this->exam->title;
        if ($this->title) {
            $title .= " - {$this->title}";
        }
        return $title;
    }

    /**
     * Get participant count for this session.
     */
    public function getParticipantCountAttribute(): int
    {
        return $this->attempts()
            ->select('candidate_type', 'candidate_id')
            ->distinct()
            ->count();
    }

    /**
     * Get completed count for this session.
     */
    public function getCompletedCountAttribute(): int
    {
        return $this->completedAttempts()->count();
    }

    /**
     * Get average score for this session.
     */
    public function getAverageScoreAttribute(): ?float
    {
        return $this->completedAttempts()->avg('score_percentage');
    }

    /**
     * Get pass rate for this session.
     */
    public function getPassRateAttribute(): ?float
    {
        $completed = $this->completedAttempts()->count();
        if ($completed === 0) {
            return null;
        }

        $passed = $this->completedAttempts()->where('passed', true)->count();
        return ($passed / $completed) * 100;
    }
}
