<?php

namespace App\Models;

use App\Enums\DifficultyLevel;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Validation\Rule;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class QuestionBank extends Model
{
    use HasFactory, SoftDeletes, LogsActivity;

    // Constants for validation
    public const QUESTION_TYPES = [
        'single_select',
        'true_false',
        'essay',
    ];

    public const TARGET_AUDIENCES = [
        'student',
        'staff',
        'applicant',
    ];

    public const DIFFICULTY_LEVELS = [
        'easy',
        'medium',
        'hard',
    ];

    protected $fillable = [
        'category_id',
        'question_text',
        'question_type',
        'difficulty',
        'subject',
        'target_audience',
        'session',
        'department',
        'options',
        'correct_answer',
        'min_words',
        'max_words',
        'explanation',
        'time_limit',
        'is_active',
        'reference',
        'import_batch_id',
        'imported_at',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'options' => 'array',
        'imported_at' => 'datetime',
        'is_active' => 'boolean',
        'time_limit' => 'integer',
        'min_words' => 'integer',
        'max_words' => 'integer',
        'difficulty' => DifficultyLevel::class,
    ];

    protected $attributes = [
        'question_type' => 'single_select',
        'target_audience' => 'student',
        'difficulty' => 'medium',
        'marks' => 1,
        'is_active' => true,
    ];

    // Validation rules
    public static function validationRules(array $extra = []): array
    {
        return array_merge([
            'question_text' => 'required|string|min:10|max:2000',
            'question_type' => ['required', 'string', Rule::in(self::QUESTION_TYPES)],
            'subject' => 'nullable|string|max:255',
            'target_audience' => ['required', 'string', Rule::in(self::TARGET_AUDIENCES)],
            'session' => 'nullable|string|max:255',
            'department' => 'nullable|string|max:255',
            'options' => 'required|array|min:2',
            'correct_answer' => 'required|string|max:10',
            'explanation' => 'nullable|string|max:1000',
            'marks' => 'required|integer|min:1|max:100',
            'time_limit' => 'nullable|integer|min:1|max:3600',
            'is_active' => 'boolean',
            'reference' => 'nullable|string|max:500',
        ], $extra);
    }

    // Activity Log Configuration
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['question_text', 'subject', 'target_audience', 'session', 'is_active'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    // Relationships
    public function category(): BelongsTo
    {
        return $this->belongsTo(QuestionCategory::class, 'category_id');
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function usageStats(): HasOne
    {
        return $this->hasOne(QuestionUsageStat::class);
    }

    public function examAnswers(): HasMany
    {
        return $this->hasMany(ExamAnswer::class, 'question_bank_id');
    }

    // Scopes for filtering
    public function scopeForStudents(Builder $query): Builder
    {
        return $query->where('target_audience', 'student');
    }

    public function scopeForStaff(Builder $query): Builder
    {
        return $query->where('target_audience', 'staff');
    }

    public function scopeForSession(Builder $query, string $session): Builder
    {
        return $query->where('session', $session);
    }

    public function scopeActive(Builder $query): Builder
    {
        return $query->where('is_active', true);
    }

    public function scopeBySubject(Builder $query, string $subject): Builder
    {
        return $query->where('subject', $subject);
    }

    public function scopeByDepartment(Builder $query, string $department): Builder
    {
        return $query->where('department', $department);
    }

    public function scopeMcq(Builder $query): Builder
    {
        return $query->where('question_type', 'single_select');
    }

    public function scopeTrueFalse(Builder $query): Builder
    {
        return $query->where('question_type', 'true_false');
    }

    public function scopeByDifficulty(Builder $query, string|DifficultyLevel $difficulty): Builder
    {
        $value = $difficulty instanceof DifficultyLevel ? $difficulty->value : $difficulty;
        return $query->where('difficulty', $value);
    }

    public function scopeByCategory(Builder $query, int $categoryId): Builder
    {
        return $query->where('category_id', $categoryId);
    }

    public function scopeSearch(Builder $query, string $search): Builder
    {
        return $query->where(function ($q) use ($search) {
            $q->where('question_text', 'like', "%{$search}%")
              ->orWhere('subject', 'like', "%{$search}%");
        });
    }

    // Utility methods
    public function getFormattedOptionsAttribute(): array
    {
        if ($this->question_type === 'true_false') {
            return [
                'A' => 'True',
                'B' => 'False'
            ];
        }

        $options = $this->options ?? [];
        
        // Handle empty options
        if (empty($options)) {
            return [];
        }

        // Check if options is in the new format: [{key: "A", text: "..."}, ...]
        // Convert to the expected format: ["A" => "...", "B" => "..."]
        if (isset($options[0]) && is_array($options[0]) && isset($options[0]['key'], $options[0]['text'])) {
            $formatted = [];
            foreach ($options as $option) {
                $key = strtoupper($option['key'] ?? '');
                $formatted[$key] = $option['text'] ?? '';
            }
            return $formatted;
        }

        // Already in correct format: {a: "Heart", b: "Brain", ...}
        // Normalize keys to uppercase
        $formatted = [];
        foreach ($options as $key => $text) {
            $formatted[strtoupper($key)] = is_string($text) ? $text : (string) $text;
        }
        return $formatted;
    }

    public function getCorrectOptionTextAttribute(): string
    {
        $options = $this->formatted_options;
        return $options[strtoupper($this->correct_answer)] ?? $options[$this->correct_answer] ?? 'N/A';
    }

    public function isCorrectAnswer(string $answer): bool
    {
        return strcasecmp($this->correct_answer, $answer) === 0;
    }

    /**
     * Check if this is an essay question.
     */
    public function isEssay(): bool
    {
        return $this->question_type === 'essay';
    }

    /**
     * Check if this question requires manual grading.
     */
    public function requiresManualGrading(): bool
    {
        return $this->isEssay();
    }

    public function getQuestionPreview(int $length = 100): string
    {
        return strlen($this->question_text) > $length 
            ? substr($this->question_text, 0, $length) . '...'
            : $this->question_text;
    }

    public function getTargetAudienceBadgeClass(): string
    {
        return match($this->target_audience) {
            'student' => 'bg-blue-100 text-blue-800',
            'staff' => 'bg-purple-100 text-purple-800',
            default => 'bg-gray-100 text-gray-800'
        };
    }

    public function getQuestionTypeBadgeClass(): string
    {
        return match($this->question_type) {
            'single_select' => 'bg-indigo-100 text-indigo-800',
            'true_false' => 'bg-cyan-100 text-cyan-800',
            'essay' => 'bg-amber-100 text-amber-800',
            default => 'bg-gray-100 text-gray-800'
        };
    }

    // Static helper methods
    public static function getAvailableSessions(): array
    {
        return self::distinct()
            ->whereNotNull('session')
            ->pluck('session')
            ->sort()
            ->values()
            ->toArray();
    }

    public static function getAvailableSubjects(): array
    {
        return self::distinct()
            ->whereNotNull('subject')
            ->pluck('subject')
            ->sort()
            ->values()
            ->toArray();
    }

    public static function getAvailableDepartments(): array
    {
        return self::distinct()
            ->whereNotNull('department')
            ->pluck('department')
            ->sort()
            ->values()
            ->toArray();
    }

    // Validation helpers
    public static function getQuestionTypeOptions(): array
    {
        return [
            'single_select' => 'Multiple Choice (Single Select)',
            'true_false' => 'True/False',
            'essay' => 'Essay/Written Answer',
        ];
    }

    public static function getTargetAudienceOptions(): array
    {
        $terminology = app(\App\Settings\TerminologySettings::class);
        
        return [
            'student' => $terminology->primary_user_singular,
            'staff' => $terminology->secondary_user_singular,
            'applicant' => 'Applicant',
        ];
    }

    public static function getDifficultyOptions(): array
    {
        return [
            'easy' => 'Easy',
            'medium' => 'Medium',
            'hard' => 'Hard',
        ];
    }

    public static function getCategoryOptions(): array
    {
        return QuestionCategory::active()
            ->ordered()
            ->pluck('name', 'id')
            ->toArray();
    }

    // Performance optimization methods
    
    /**
     * Get cached count of questions by target audience
     */
    public static function getCachedCountByAudience(string $audience): int
    {
        return cache()->remember(
            "question_bank_count_{$audience}",
            now()->addMinutes(10),
            fn() => static::where('target_audience', $audience)
                ->where('is_active', true)
                ->count()
        );
    }

    /**
     * Get cached questions for a specific session with optimized loading
     */
    public static function getCachedSessionQuestions(string $session, int $limit = 50): \Illuminate\Database\Eloquent\Collection
    {
        return cache()->remember(
            "session_questions_{$session}_{$limit}",
            now()->addMinutes(5),
            fn() => static::with(['creator:id,name'])
                ->forStudents()
                ->forSession($session)
                ->active()
                ->limit($limit)
                ->get(['id', 'question_text', 'question_type', 'options', 'correct_answer', 'marks', 'time_limit', 'created_by'])
        );
    }

    /**
     * Bust cache when model is updated
     */
    protected static function booted(): void
    {
        static::saved(function ($questionBank) {
            cache()->forget("question_bank_count_{$questionBank->target_audience}");
            if ($questionBank->session) {
                cache()->forget("session_questions_{$questionBank->session}_50");
            }
        });

        static::deleted(function ($questionBank) {
            cache()->forget("question_bank_count_{$questionBank->target_audience}");
            if ($questionBank->session) {
                cache()->forget("session_questions_{$questionBank->session}_50");
            }
        });
    }
}
