<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class QuestionUsageStat extends Model
{
    use HasFactory;

    protected $fillable = [
        'question_bank_id',
        'times_used',
        'times_correct',
        'times_incorrect',
        'success_rate',
        'average_time_taken',
        'last_used_at',
        'last_updated_stats',
    ];

    protected $casts = [
        'times_used' => 'integer',
        'times_correct' => 'integer',
        'times_incorrect' => 'integer',
        'success_rate' => 'decimal:2',
        'average_time_taken' => 'decimal:2',
        'last_used_at' => 'datetime',
        'last_updated_stats' => 'datetime',
    ];

    // Relationships
    public function questionBank(): BelongsTo
    {
        return $this->belongsTo(QuestionBank::class);
    }

    // Utility methods
    public function updateStats(bool $wasCorrect, float $timeTaken = null): void
    {
        $this->increment('times_used');
        
        if ($wasCorrect) {
            $this->increment('times_correct');
        } else {
            $this->increment('times_incorrect');
        }

        // Calculate success rate
        $this->success_rate = $this->times_used > 0 
            ? round(($this->times_correct / $this->times_used) * 100, 2)
            : 0;

        // Update average time taken
        if ($timeTaken !== null) {
            $this->average_time_taken = $this->times_used > 1
                ? round((($this->average_time_taken * ($this->times_used - 1)) + $timeTaken) / $this->times_used, 2)
                : $timeTaken;
        }

        $this->last_used_at = now();
        $this->last_updated_stats = now();
        $this->save();
    }

    public function getSuccessRateColorClass(): string
    {
        if ($this->success_rate >= 80) {
            return 'text-emerald-600';
        } elseif ($this->success_rate >= 60) {
            return 'text-amber-600';
        } else {
            return 'text-red-600';
        }
    }

    public function getDifficultyIndicator(): string
    {
        if ($this->success_rate >= 80) {
            return 'Easy';
        } elseif ($this->success_rate >= 60) {
            return 'Medium';
        } else {
            return 'Hard';
        }
    }
}
