<?php

namespace App\Providers\Filament;

use App\Filament\Pages\Auth\Login;
use App\Settings\BrandingSettings;
use App\Settings\OrganizationSettings;
use Filament\Http\Middleware\Authenticate;
use Filament\Http\Middleware\AuthenticateSession;
use Filament\Http\Middleware\DisableBladeIconComponents;
use Filament\Http\Middleware\DispatchServingFilamentEvent;
use Filament\Pages\Dashboard;
use Filament\Panel;
use Filament\PanelProvider;
use Filament\Support\Colors\Color;
use Filament\Support\Facades\FilamentView;
use Filament\View\PanelsRenderHook;
use Illuminate\Cookie\Middleware\AddQueuedCookiesToResponse;
use Illuminate\Cookie\Middleware\EncryptCookies;
use Illuminate\Foundation\Http\Middleware\VerifyCsrfToken;
use Illuminate\Routing\Middleware\SubstituteBindings;
use Illuminate\Session\Middleware\StartSession;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\Middleware\ShareErrorsFromSession;

class AdminPanelProvider extends PanelProvider
{
    public function panel(Panel $panel): Panel
    {
        // Default values
        $brandName = 'CBT System';
        $brandLogo = null;
        $brandFavicon = null;
        $primaryPalette = Color::Blue;
        $secondaryPalette = Color::Gray;

        // Only load settings if the settings table exists (not during migration)
        if (Schema::hasTable('settings')) {
            try {
                $organizationSettings = app(OrganizationSettings::class);
                $brandingSettings = app(BrandingSettings::class);

                // Get dynamic brand name and logo
                $brandName = $organizationSettings->name ?? 'CBT System';
                $brandLogo = $brandingSettings->logo_path
                    ? Storage::disk('public')->url($brandingSettings->logo_path)
                    : null;
                $brandFavicon = $brandingSettings->favicon_path
                    ? Storage::disk('public')->url($brandingSettings->favicon_path)
                    : $brandLogo; // Fall back to logo if no favicon set

                // Get colors from settings and generate Filament-compatible palette
                $primaryPalette = $brandingSettings->getFilamentPrimaryColor();
                $secondaryPalette = $brandingSettings->getFilamentSecondaryColor();
            } catch (\Exception $e) {
                // Settings not ready yet, use defaults
            }
        }

        $panel = $panel
            ->default()
            ->id('admin')
            ->path('admin')
            ->login(\App\Filament\Pages\Auth\Login::class)
            ->loginRouteSlug('login')
            ->brandName($brandName . ' - Admin')
            ->darkMode(false)
            ->colors([
                'primary' => $primaryPalette,
                'secondary' => $secondaryPalette,
            ])
            ->discoverResources(in: app_path('Filament/Resources'), for: 'App\Filament\Resources')
            ->discoverPages(in: app_path('Filament/Pages'), for: 'App\Filament\Pages')
            ->pages([
                Dashboard::class,
            ])
            ->discoverWidgets(in: app_path('Filament/Widgets'), for: 'App\Filament\Widgets')
            ->widgets([
                \App\Filament\Widgets\CBTStatsOverview::class,
                \App\Filament\Widgets\ExamPerformanceChart::class,
                \App\Filament\Widgets\ActiveExamSessions::class,
                \App\Filament\Widgets\RecentExamAttempts::class,
            ])
            ->middleware([
                EncryptCookies::class,
                AddQueuedCookiesToResponse::class,
                StartSession::class,
                AuthenticateSession::class,
                ShareErrorsFromSession::class,
                VerifyCsrfToken::class,
                SubstituteBindings::class,
                DisableBladeIconComponents::class,
                DispatchServingFilamentEvent::class,
            ])
            ->authMiddleware([
                Authenticate::class,
            ])
            ->renderHook(
                PanelsRenderHook::HEAD_END,
                fn () => $this->getLoginBackgroundStyles(),
                scopes: Login::class,
            );

        // Apply logo if set
        if ($brandLogo) {
            $panel
                ->brandLogo($brandLogo)
                ->brandLogoHeight('3rem');
        }

        // Apply favicon if set
        if ($brandFavicon ?? null) {
            $panel->favicon($brandFavicon);
        }

        return $panel;
    }

    /**
     * Generate CSS styles for login page background image.
     */
    protected function getLoginBackgroundStyles(): string
    {
        $backgroundUrl = asset('images/fnph-building.jpg'); // Default

        if (Schema::hasTable('settings')) {
            try {
                $brandingSettings = app(BrandingSettings::class);
                $backgroundUrl = $brandingSettings->getAuthBackgroundUrl();
            } catch (\Exception $e) {
                // Use default
            }
        }

        return <<<HTML
        <style>
            /* Full-page background for login with gradient overlay */
            .fi-simple-layout {
                background-image: 
                    linear-gradient(to bottom right, rgba(30, 58, 138, 0.7), rgba(30, 64, 175, 0.6), rgba(88, 28, 135, 0.7)),
                    url('{$backgroundUrl}');
                background-size: cover, cover;
                background-position: center, center;
                background-repeat: no-repeat, no-repeat;
                background-attachment: fixed, fixed;
            }
            
            /* Make the form card semi-transparent with blur */
            .fi-simple-main {
                background: rgba(255, 255, 255, 0.95) !important;
                backdrop-filter: blur(10px);
                box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25) !important;
            }
            
            /* Dark mode support */
            .dark .fi-simple-main {
                background: rgba(17, 24, 39, 0.95) !important;
            }
        </style>
        HTML;
    }
}
