<?php

namespace App\Services;

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpWord\PhpWord;
use PhpOffice\PhpWord\IOFactory;

class TemplateGeneratorService
{
    /**
     * Generate Excel template for question import
     */
    public function generateExcelTemplate(): string
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Set headers
        $headers = [
            'A1' => 'Question',
            'B1' => 'Type',
            'C1' => 'Option A',
            'D1' => 'Option B',
            'E1' => 'Option C',
            'F1' => 'Option D',
            'G1' => 'Correct Answer',
            'H1' => 'Explanation',
        ];
        
        // Style the headers
        foreach ($headers as $cell => $header) {
            $sheet->setCellValue($cell, $header);
            $sheet->getStyle($cell)->getFont()->setBold(true);
            $sheet->getStyle($cell)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setRGB('E3F2FD');
            $sheet->getStyle($cell)->getAlignment()
                ->setHorizontal(Alignment::HORIZONTAL_CENTER);
        }
        
        // Set column widths
        $sheet->getColumnDimension('A')->setWidth(50);
        $sheet->getColumnDimension('B')->setWidth(15);
        $sheet->getColumnDimension('C')->setWidth(30);
        $sheet->getColumnDimension('D')->setWidth(30);
        $sheet->getColumnDimension('E')->setWidth(30);
        $sheet->getColumnDimension('F')->setWidth(30);
        $sheet->getColumnDimension('G')->setWidth(15);
        $sheet->getColumnDimension('H')->setWidth(40);
        
        // Add sample data rows (MCQ and True/False examples)
        $sampleData = [
            // MCQ example
            [
                'What is the capital of Nigeria?',
                'single_select',
                'Lagos',
                'Abuja',
                'Kano',
                'Port Harcourt',
                'b',
                'Abuja became the capital of Nigeria in 1991.',
            ],
            // True/False example
            [
                'The heart has four chambers.',
                'true_false',
                '',
                '',
                '',
                '',
                'true',
                'The heart consists of two atria and two ventricles.',
            ],
        ];
        
        $row = 2;
        foreach ($sampleData as $rowData) {
            $col = 'A';
            foreach ($rowData as $value) {
                $sheet->setCellValue($col . $row, $value);
                $col++;
            }
            $row++;
        }
        
        // Add instructions sheet
        $instructionsSheet = $spreadsheet->createSheet();
        $instructionsSheet->setTitle('Instructions');
        
        $instructions = [
            'QUESTION IMPORT TEMPLATE INSTRUCTIONS',
            '',
            'Column Descriptions:',
            '• Question: The question text (required)',
            '• Type: Question type - "single_select" or "true_false" (optional, default: single_select)',
            '• Option A-D: Multiple choice options (required for single_select questions)',
            '• Correct Answer: The correct answer key (required)',
            '  - For single_select: use "a", "b", "c", or "d"',
            '  - For true_false: use "true" or "false"',
            '• Explanation: Optional explanation for the answer',
            '',
            'Note: Subject is selected during the import process, not in this file.',
            '',
            'Tips:',
            '• Delete sample data before importing your questions',
            '• Ensure each question has a valid correct answer',
            '• For true/false questions, leave Option A-D columns empty',
            '• Maximum file size: 10MB',
            '• Supported formats: .xlsx, .xls, .csv',
            '',
            'For questions, contact the system administrator.'
        ];
        
        $row = 1;
        foreach ($instructions as $instruction) {
            $instructionsSheet->setCellValue('A' . $row, $instruction);
            if ($row === 1) {
                $instructionsSheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(14);
            }
            $row++;
        }
        
        $instructionsSheet->getColumnDimension('A')->setWidth(80);
        
        // Set active sheet back to the main sheet
        $spreadsheet->setActiveSheetIndex(0);
        
        // Save to temporary file
        $tempFile = tempnam(sys_get_temp_dir(), 'question_template_') . '.xlsx';
        $writer = new Xlsx($spreadsheet);
        $writer->save($tempFile);
        
        return $tempFile;
    }
    
    /**
     * Generate Word template for question import
     */
    public function generateWordTemplate(): string
    {
        $phpWord = new PhpWord();
        
        // Add title
        $section = $phpWord->addSection();
        $section->addTitle('Question Import Template', 1);
        
        $section->addText(
            'Instructions: Format your questions as shown below. Each question should start with a number, followed by the question text. For multiple choice questions, list options as a), b), c), d). Mark the correct answer with an asterisk (*) or write "(correct)" after it.',
            ['size' => 11]
        );
        
        $section->addTextBreak();
        
        $section->addText(
            'IMPORTANT: Each element (question, options, answer, explanation) must be on its own separate line. Press Enter after each line - do not use Shift+Enter or combine multiple items on the same line.',
            ['size' => 11, 'bold' => true, 'color' => 'CC0000']
        );
        
        $section->addTextBreak(2);
        
        // Add sample questions
        $section->addTitle('Sample Questions:', 2);
        
        // Multiple choice example
        $section->addTextBreak();
        $section->addText('1. What is the largest organ in the human body?', ['bold' => true]);
        $section->addText('a) Heart');
        $section->addText('b) Brain');
        $section->addText('c) Skin *', ['bold' => true]); // Correct answer marked
        $section->addText('d) Liver');
        $section->addText('Explanation: The skin is the largest organ by surface area and weight.', ['italic' => true]);
        
        $section->addTextBreak();
        
        // True/False example
        $section->addText('2. The heart has four chambers. (True/False)', ['bold' => true]);
        $section->addText('Answer: True *', ['bold' => true]);
        $section->addText('Explanation: The heart consists of two atria and two ventricles.', ['italic' => true]);
        
        $section->addTextBreak();
        
        // Another multiple choice
        $section->addText('3. Which vitamin is produced when skin is exposed to sunlight?', ['bold' => true]);
        $section->addText('a) Vitamin A');
        $section->addText('b) Vitamin B12');
        $section->addText('c) Vitamin C');
        $section->addText('d) Vitamin D (correct)', ['bold' => true]);
        $section->addText('Explanation: Vitamin D is synthesized in the skin when exposed to UVB radiation.', ['italic' => true]);
        
        $section->addTextBreak(2);
        
        // Add formatting instructions
        $section->addTitle('Formatting Guidelines:', 2);
        $instructionsList = [
            'IMPORTANT: Each line must be a separate paragraph (press Enter, not Shift+Enter)',
            'Start each question with a number followed by a period or parenthesis',
            'For multiple choice: Use a), b), c), d) or a., b., c., d.',
            'Mark correct answers with * or (correct)',
            'For True/False questions, put the answer on a separate line: "Answer: True *"',
            'Add explanations on a new line starting with "Explanation:"',
            'Do NOT combine question text and answer on the same line',
            'Questions can be on any medical topic relevant to your course',
            'Ensure questions are clear and unambiguous'
        ];
        
        foreach ($instructionsList as $instruction) {
            $section->addListItem($instruction);
        }
        
        $section->addTextBreak();
        $section->addText(
            'Note: After creating your questions in this format, save the document and upload it through the bulk import feature in the admin panel.',
            ['italic' => true, 'color' => '666666']
        );
        
        // Save to temporary file
        $tempFile = tempnam(sys_get_temp_dir(), 'question_template_') . '.docx';
        $writer = IOFactory::createWriter($phpWord, 'Word2007');
        $writer->save($tempFile);
        
        return $tempFile;
    }
}