<?php

namespace App\Settings;

use Spatie\LaravelSettings\Settings;

class TerminologySettings extends Settings
{
    // Primary user terminology (exam takers)
    public string $primary_user_singular;
    public string $primary_user_plural;
    public string $primary_user_id_label;
    public string $primary_user_id_field;

    // Secondary user terminology (staff/examiners)
    public string $secondary_user_singular;
    public string $secondary_user_plural;
    public string $secondary_user_id_label;
    public string $secondary_user_id_field;

    // Applicant user terminology (recruitment candidates)
    public ?string $applicant_user_singular;
    public ?string $applicant_user_plural;
    public ?string $applicant_id_label;
    public ?string $applicant_id_field;

    // Academic/period terminology
    public string $period_label;
    public string $period_format;

    // Organizational unit terminology
    public string $division_label;

    public static function group(): string
    {
        return 'terminology';
    }

    /**
     * Get the primary user label (singular or plural).
     */
    public function getPrimaryUserLabel(bool $plural = false): string
    {
        return $plural ? $this->primary_user_plural : $this->primary_user_singular;
    }

    /**
     * Get the secondary user label (singular or plural).
     */
    public function getSecondaryUserLabel(bool $plural = false): string
    {
        return $plural ? $this->secondary_user_plural : $this->secondary_user_singular;
    }

    /**
     * Get the applicant user label (singular or plural).
     */
    public function getApplicantUserLabel(bool $plural = false): string
    {
        if ($plural) {
            return $this->applicant_user_plural ?? 'Applicants';
        }
        return $this->applicant_user_singular ?? 'Applicant';
    }

    /**
     * Apply preset terminology from config.
     */
    public function applyPreset(string $presetKey): void
    {
        $preset = config("cbt.user_type_presets.{$presetKey}");

        if (! $preset) {
            return;
        }

        $this->primary_user_singular = $preset['primary']['singular'];
        $this->primary_user_plural = $preset['primary']['plural'];
        $this->primary_user_id_label = $preset['primary']['id_label'];

        $this->secondary_user_singular = $preset['secondary']['singular'];
        $this->secondary_user_plural = $preset['secondary']['plural'];
        $this->secondary_user_id_label = $preset['secondary']['id_label'];

        $this->period_label = $preset['period_label'];
        $this->division_label = $preset['division_label'];

        // Apply applicant settings if present in preset
        if (isset($preset['applicant'])) {
            $this->applicant_user_singular = $preset['applicant']['singular'];
            $this->applicant_user_plural = $preset['applicant']['plural'];
            $this->applicant_id_label = $preset['applicant']['id_label'];
        }

        $this->save();
    }

    /**
     * Get all available presets for selection.
     */
    public static function getAvailablePresets(): array
    {
        $presets = config('cbt.user_type_presets', []);

        return collect($presets)->map(fn ($preset, $key) => [
            'key' => $key,
            'label' => $preset['label'],
            'description' => $preset['description'],
        ])->values()->all();
    }
}
