<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('question_banks', function (Blueprint $table) {
            $table->id();
            
            // Basic question information
            $table->text('question_text');
            $table->string('question_type')->default('single_select'); // single_select, true_false
            $table->string('subject')->nullable();
            $table->string('topic')->nullable();
            
            // Category and target audience
            $table->string('target_audience')->default('student'); // student, staff
            $table->string('session')->nullable(); // Academic session for students
            $table->string('department')->nullable(); // Department relevance
            
            // MCQ Options (JSON format for flexibility)
            $table->json('options'); // Array of options for single_select, or true/false for true_false
            $table->string('correct_answer'); // Store the correct option key or true/false
            $table->text('explanation')->nullable(); // Optional explanation for the answer
            
            // Question metadata
            $table->integer('marks')->default(1); // Points awarded for correct answer
            $table->integer('time_limit')->nullable(); // Time limit in seconds (optional)
            $table->boolean('is_active')->default(true);
            $table->string('reference')->nullable(); // Reference material or source
            
            // Import tracking
            $table->string('import_batch_id')->nullable(); // Track bulk imports
            $table->timestamp('imported_at')->nullable();
            
            // Audit fields
            $table->foreignId('created_by')->nullable()->constrained('users')->onDelete('set null');
            $table->foreignId('updated_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamps();
            $table->softDeletes(); // Add soft deletes support
            
            // Performance indexes
            $table->index(['target_audience', 'session', 'is_active']);
            $table->index(['subject', 'question_type']);
            $table->index(['department', 'target_audience']);
            $table->index('import_batch_id');
        });

        // Create DB-specific fulltext / search indexes
        if (DB::getDriverName() === 'mysql') {
            Schema::table('question_banks', function (Blueprint $table) {
                $table->fullText(['question_text', 'subject', 'topic']); // MySQL native fulltext
            });
        }

        if (DB::getDriverName() === 'pgsql') {
            // Postgres: create a GIN index using to_tsvector for full-text search
            DB::statement("CREATE INDEX question_banks_search_idx ON question_banks USING GIN(to_tsvector('english', coalesce(question_text,'') || ' ' || coalesce(subject,'') || ' ' || coalesce(topic,'')))");
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('question_banks');
    }
};
