<?php

namespace Database\Seeders;

use App\Enums\ExamAttemptStatus;
use App\Enums\ExamSessionStatus;
use App\Models\Exam;
use App\Models\ExamAnswer;
use App\Models\ExamAttempt;
use App\Models\ExamSession;
use App\Models\QuestionBank;
use App\Models\QuestionCategory;
use App\Models\Staff;
use App\Models\Student;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class TestDataSeeder extends Seeder
{
    /**
     * Seed comprehensive test data for the CBT system.
     */
    public function run(): void
    {
        $this->command->info('Creating test data...');
        
        // Create admin user if not exists
        $admin = User::firstOrCreate(
            ['email' => 'admin@fnph.edu.ng'],
            [
                'name' => 'System Administrator',
                'password' => 'password',
            ]
        );
        
        $this->command->info('✓ Admin user created');
        
        // Create question categories
        $categories = $this->createCategories();
        $this->command->info('✓ Question categories created');
        
        // Create questions
        $questions = $this->createQuestions($categories, $admin);
        $this->command->info('✓ Questions created: ' . count($questions));
        
        // Create students
        $students = $this->createStudents();
        $this->command->info('✓ Students created: ' . count($students));
        
        // Create staff
        $staff = $this->createStaff();
        $this->command->info('✓ Staff created: ' . count($staff));
        
        // Create exams
        $exams = $this->createExams($questions, $admin);
        $this->command->info('✓ Exams created: ' . count($exams));
        
        // Create exam sessions
        $sessions = $this->createExamSessions($exams, $admin);
        $this->command->info('✓ Exam sessions created: ' . count($sessions));
        
        // Create some exam attempts (completed results)
        $attempts = $this->createExamAttempts($sessions, $students, $questions);
        $this->command->info('✓ Exam attempts created: ' . count($attempts));
        
        $this->command->info('');
        $this->command->info('=== Test Data Summary ===');
        $this->command->info("Admin: admin@fnph.edu.ng / password");
        $this->command->info("Students: STU/2024/0001 - STU/2024/0010 / password");
        $this->command->info("Staff: STF/0001 - STF/0005 / password");
        $this->command->info('');
    }
    
    private function createCategories(): array
    {
        $categoriesData = [
            ['name' => 'Nursing Fundamentals', 'slug' => 'nursing-fundamentals', 'description' => 'Basic nursing concepts and practices'],
            ['name' => 'Medical-Surgical Nursing', 'slug' => 'medical-surgical', 'description' => 'Care of patients with medical and surgical conditions'],
            ['name' => 'Pharmacology', 'slug' => 'pharmacology', 'description' => 'Drug actions, interactions, and administration'],
            ['name' => 'Anatomy & Physiology', 'slug' => 'anatomy-physiology', 'description' => 'Human body structure and function'],
            ['name' => 'Mental Health Nursing', 'slug' => 'mental-health', 'description' => 'Psychiatric and mental health nursing'],
            ['name' => 'Community Health', 'slug' => 'community-health', 'description' => 'Public health and community nursing'],
        ];
        
        $categories = [];
        foreach ($categoriesData as $index => $data) {
            $categories[] = QuestionCategory::firstOrCreate(
                ['slug' => $data['slug']],
                array_merge($data, ['is_active' => true, 'sort_order' => $index + 1])
            );
        }
        
        return $categories;
    }
    
    private function createQuestions(array $categories, User $admin): array
    {
        $questionsData = [
            // Nursing Fundamentals
            [
                'category' => 'nursing-fundamentals',
                'question_text' => 'What is the normal range for adult body temperature in Celsius?',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => '35.0 - 36.0°C'],
                    ['key' => 'B', 'text' => '36.1 - 37.2°C'],
                    ['key' => 'C', 'text' => '37.5 - 38.5°C'],
                    ['key' => 'D', 'text' => '38.5 - 39.5°C'],
                ],
                'correct_answer' => 'B',
                'explanation' => 'Normal adult body temperature ranges from 36.1°C to 37.2°C (97°F to 99°F).',
                'subject' => 'Vital Signs',
            ],
            [
                'category' => 'nursing-fundamentals',
                'question_text' => 'Which position is best for a patient with dyspnea (difficulty breathing)?',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => 'Supine position'],
                    ['key' => 'B', 'text' => 'Prone position'],
                    ['key' => 'C', 'text' => 'Fowler\'s position'],
                    ['key' => 'D', 'text' => 'Trendelenburg position'],
                ],
                'correct_answer' => 'C',
                'explanation' => 'Fowler\'s position (sitting up at 45-60 degrees) helps expand the lungs and makes breathing easier.',
                'subject' => 'Patient Positioning',
            ],
            [
                'category' => 'nursing-fundamentals',
                'question_text' => 'Hand hygiene is the single most important measure to prevent infection.',
                'question_type' => 'true_false',
                'options' => [
                    ['key' => 'A', 'text' => 'True'],
                    ['key' => 'B', 'text' => 'False'],
                ],
                'correct_answer' => 'A',
                'explanation' => 'Hand hygiene is recognized as the most effective way to prevent the spread of infections in healthcare settings.',
                'subject' => 'Infection Control',
            ],
            [
                'category' => 'nursing-fundamentals',
                'question_text' => 'What is the normal adult respiratory rate per minute?',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => '8-10 breaths'],
                    ['key' => 'B', 'text' => '12-20 breaths'],
                    ['key' => 'C', 'text' => '22-28 breaths'],
                    ['key' => 'D', 'text' => '30-40 breaths'],
                ],
                'correct_answer' => 'B',
                'explanation' => 'Normal adult respiratory rate is 12-20 breaths per minute at rest.',
                'subject' => 'Vital Signs',
            ],
            
            // Medical-Surgical Nursing
            [
                'category' => 'medical-surgical',
                'question_text' => 'A patient with diabetes mellitus should be monitored for which of the following?',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => 'Blood glucose levels only'],
                    ['key' => 'B', 'text' => 'Blood glucose, kidney function, and foot care'],
                    ['key' => 'C', 'text' => 'Liver function only'],
                    ['key' => 'D', 'text' => 'Cardiac enzymes only'],
                ],
                'correct_answer' => 'B',
                'explanation' => 'Diabetes affects multiple body systems. Monitoring includes blood glucose, kidney function (nephropathy), and foot care (neuropathy).',
                'subject' => 'Endocrine Disorders',
            ],
            [
                'category' => 'medical-surgical',
                'question_text' => 'The most common site for a myocardial infarction is the:',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => 'Right atrium'],
                    ['key' => 'B', 'text' => 'Left ventricle'],
                    ['key' => 'C', 'text' => 'Right ventricle'],
                    ['key' => 'D', 'text' => 'Left atrium'],
                ],
                'correct_answer' => 'B',
                'explanation' => 'The left ventricle is the most common site for MI due to the high oxygen demand and workload of pumping blood to the systemic circulation.',
                'subject' => 'Cardiovascular Disorders',
            ],
            [
                'category' => 'medical-surgical',
                'question_text' => 'Post-operative patients should be encouraged to do deep breathing exercises.',
                'question_type' => 'true_false',
                'options' => [
                    ['key' => 'A', 'text' => 'True'],
                    ['key' => 'B', 'text' => 'False'],
                ],
                'correct_answer' => 'A',
                'explanation' => 'Deep breathing exercises help prevent atelectasis (lung collapse) and pneumonia, which are common post-operative complications.',
                'subject' => 'Post-Operative Care',
            ],
            
            // Pharmacology
            [
                'category' => 'pharmacology',
                'question_text' => 'Which medication is commonly used as an antidote for acetaminophen overdose?',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => 'Naloxone'],
                    ['key' => 'B', 'text' => 'N-acetylcysteine'],
                    ['key' => 'C', 'text' => 'Flumazenil'],
                    ['key' => 'D', 'text' => 'Atropine'],
                ],
                'correct_answer' => 'B',
                'explanation' => 'N-acetylcysteine (NAC) is the antidote for acetaminophen (paracetamol) overdose, helping to protect the liver from damage.',
                'subject' => 'Antidotes',
            ],
            [
                'category' => 'pharmacology',
                'question_text' => 'Warfarin should be taken with Vitamin K-rich foods for maximum effect.',
                'question_type' => 'true_false',
                'options' => [
                    ['key' => 'A', 'text' => 'True'],
                    ['key' => 'B', 'text' => 'False'],
                ],
                'correct_answer' => 'B',
                'explanation' => 'Vitamin K antagonizes warfarin\'s anticoagulant effect. Patients on warfarin should maintain consistent Vitamin K intake.',
                'subject' => 'Anticoagulants',
            ],
            [
                'category' => 'pharmacology',
                'question_text' => 'The "Five Rights" of medication administration include all EXCEPT:',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => 'Right patient'],
                    ['key' => 'B', 'text' => 'Right documentation'],
                    ['key' => 'C', 'text' => 'Right weather'],
                    ['key' => 'D', 'text' => 'Right time'],
                ],
                'correct_answer' => 'C',
                'explanation' => 'The Five Rights are: Right patient, Right drug, Right dose, Right route, and Right time. Documentation is sometimes added as a sixth right.',
                'subject' => 'Medication Safety',
            ],
            
            // Anatomy & Physiology
            [
                'category' => 'anatomy-physiology',
                'question_text' => 'The largest organ in the human body is the:',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => 'Liver'],
                    ['key' => 'B', 'text' => 'Brain'],
                    ['key' => 'C', 'text' => 'Skin'],
                    ['key' => 'D', 'text' => 'Small intestine'],
                ],
                'correct_answer' => 'C',
                'explanation' => 'The skin is the largest organ, covering approximately 2 square meters in adults and weighing about 3.6 kg.',
                'subject' => 'Integumentary System',
            ],
            [
                'category' => 'anatomy-physiology',
                'question_text' => 'Red blood cells are produced in the bone marrow.',
                'question_type' => 'true_false',
                'options' => [
                    ['key' => 'A', 'text' => 'True'],
                    ['key' => 'B', 'text' => 'False'],
                ],
                'correct_answer' => 'A',
                'explanation' => 'Red blood cells (erythrocytes) are produced in the red bone marrow through a process called erythropoiesis.',
                'subject' => 'Hematology',
            ],
            [
                'category' => 'anatomy-physiology',
                'question_text' => 'Which chamber of the heart pumps blood to the lungs?',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => 'Left atrium'],
                    ['key' => 'B', 'text' => 'Left ventricle'],
                    ['key' => 'C', 'text' => 'Right atrium'],
                    ['key' => 'D', 'text' => 'Right ventricle'],
                ],
                'correct_answer' => 'D',
                'explanation' => 'The right ventricle pumps deoxygenated blood to the lungs via the pulmonary artery for oxygenation.',
                'subject' => 'Cardiovascular System',
            ],
            
            // Mental Health
            [
                'category' => 'mental-health',
                'question_text' => 'Therapeutic communication includes which of the following techniques?',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => 'Giving advice and opinions'],
                    ['key' => 'B', 'text' => 'Active listening and open-ended questions'],
                    ['key' => 'C', 'text' => 'Changing the subject when uncomfortable'],
                    ['key' => 'D', 'text' => 'Arguing with patient\'s perceptions'],
                ],
                'correct_answer' => 'B',
                'explanation' => 'Therapeutic communication uses active listening, open-ended questions, reflection, and clarification to build rapport and understanding.',
                'subject' => 'Therapeutic Communication',
            ],
            [
                'category' => 'mental-health',
                'question_text' => 'A patient with schizophrenia may experience hallucinations.',
                'question_type' => 'true_false',
                'options' => [
                    ['key' => 'A', 'text' => 'True'],
                    ['key' => 'B', 'text' => 'False'],
                ],
                'correct_answer' => 'A',
                'explanation' => 'Hallucinations, particularly auditory hallucinations (hearing voices), are a common positive symptom of schizophrenia.',
                'subject' => 'Psychiatric Disorders',
            ],
            
            // Community Health
            [
                'category' => 'community-health',
                'question_text' => 'Primary prevention includes:',
                'question_type' => 'single_select',
                'options' => [
                    ['key' => 'A', 'text' => 'Treatment of disease'],
                    ['key' => 'B', 'text' => 'Rehabilitation after illness'],
                    ['key' => 'C', 'text' => 'Immunization and health education'],
                    ['key' => 'D', 'text' => 'Screening for early detection'],
                ],
                'correct_answer' => 'C',
                'explanation' => 'Primary prevention aims to prevent disease before it occurs through immunization, health education, and lifestyle modifications.',
                'subject' => 'Preventive Health',
            ],
            [
                'category' => 'community-health',
                'question_text' => 'The WHO recommends exclusive breastfeeding for the first 6 months of life.',
                'question_type' => 'true_false',
                'options' => [
                    ['key' => 'A', 'text' => 'True'],
                    ['key' => 'B', 'text' => 'False'],
                ],
                'correct_answer' => 'A',
                'explanation' => 'WHO recommends exclusive breastfeeding for 6 months, with continued breastfeeding along with complementary foods up to 2 years or beyond.',
                'subject' => 'Maternal Child Health',
            ],
        ];
        
        $questions = [];
        $categoryMap = collect($categories)->keyBy('slug');
        
        foreach ($questionsData as $data) {
            $categorySlug = $data['category'];
            unset($data['category']);
            
            $category = $categoryMap[$categorySlug] ?? null;
            
            $questions[] = QuestionBank::create([
                'category_id' => $category?->id,
                'question_text' => $data['question_text'],
                'question_type' => $data['question_type'],
                'options' => $data['options'],
                'correct_answer' => strtolower($data['correct_answer']),
                'explanation' => $data['explanation'],
                'subject' => $data['subject'],
                'target_audience' => 'student',
                'session' => '2024/2025',
                'department' => 'Nursing',
                'marks' => 1,
                'is_active' => true,
                'created_by' => $admin->id,
            ]);
        }
        
        return $questions;
    }
    
    private function createStudents(): array
    {
        $students = [];
        
        for ($i = 1; $i <= 10; $i++) {
            $students[] = Student::firstOrCreate(
                ['matriculation_number' => sprintf('STU/2024/%04d', $i)],
                [
                    'full_name' => fake()->name(),
                    'email' => sprintf('student%d@fnph.edu.ng', $i),
                    'password' => 'password',
                    'session' => '2024/2025',
                    'is_active' => true,
                ]
            );
        }
        
        return $students;
    }
    
    private function createStaff(): array
    {
        $staff = [];
        
        $departments = ['Nursing', 'Administration', 'Medical Records', 'Pharmacy', 'Laboratory'];
        
        for ($i = 1; $i <= 5; $i++) {
            $staffType = fake()->randomElement(['senior', 'junior']);
            $staffNumber = Staff::generateStaffNumber($staffType);

            $staff[] = Staff::firstOrCreate(
                ['staff_number' => $staffNumber],
                [
                    'full_name' => fake()->name(),
                    'email' => sprintf('staff%d@fnph.edu.ng', $i),
                    'password' => 'password',
                    'phone' => fake()->phoneNumber(),
                    'department' => $departments[$i - 1],
                    'position' => fake()->randomElement(['Instructor', 'Supervisor', 'Manager', 'Coordinator']),
                    'staff_type' => $staffType,
                    'employment_status' => 'active',
                ]
            );
        }
        
        return $staff;
    }
    
    private function createExams(array $questions, User $admin): array
    {
        $questionIds = collect($questions)->pluck('id')->toArray();
        
        // Get essay questions if available
        $essayQuestions = QuestionBank::where('question_type', 'essay')->pluck('id')->toArray();
        
        $examsData = [
            [
                'title' => 'Nursing Fundamentals Mid-Term Exam',
                'code' => 'NF-MID-2024',
                'description' => 'Mid-term examination covering basic nursing concepts, vital signs, and patient care.',
                'instructions' => "1. Read each question carefully before answering.\n2. You have 60 minutes to complete this exam.\n3. All questions carry equal marks.\n4. No negative marking for wrong answers.\n5. You cannot go back once you submit.",
                'duration_minutes' => 60,
                'total_questions' => 10,
                'passing_score' => 50,
                'target_audience' => 'student',
                'question_type' => 'mcq',
                'question_ids' => array_slice($questionIds, 0, 10),
            ],
            [
                'title' => 'Pharmacology Practice Test',
                'code' => 'PHARM-PT-2024',
                'description' => 'Practice test for pharmacology including drug actions, interactions, and administration.',
                'instructions' => "1. This is a practice test.\n2. Time limit: 30 minutes.\n3. Review your answers before submitting.",
                'duration_minutes' => 30,
                'total_questions' => 5,
                'passing_score' => 60,
                'target_audience' => 'student',
                'question_type' => 'mcq',
                'question_ids' => array_slice($questionIds, 5, 5),
            ],
            [
                'title' => 'Comprehensive Nursing Assessment',
                'code' => 'CNA-2024',
                'description' => 'Comprehensive assessment covering all nursing subjects.',
                'instructions' => "1. This is a comprehensive exam.\n2. Time limit: 90 minutes.\n3. Answer all questions.\n4. Good luck!",
                'duration_minutes' => 90,
                'total_questions' => 15,
                'passing_score' => 50,
                'target_audience' => 'student',
                'question_type' => 'mcq',
                'question_ids' => array_slice($questionIds, 0, 15),
            ],
        ];
        
        // Add essay exam if essay questions exist
        if (!empty($essayQuestions)) {
            $examsData[] = [
                'title' => 'Healthcare Essay Assessment',
                'code' => 'ESSAY-HCA-2024',
                'description' => 'Essay-based assessment covering healthcare topics and critical thinking.',
                'instructions' => "1. Read each question carefully.\n2. Write clear and well-structured responses.\n3. Ensure your answers are within the specified word limits.\n4. Time limit: 90 minutes.\n5. Your responses will be manually graded.",
                'duration_minutes' => 90,
                'total_questions' => min(5, count($essayQuestions)),
                'passing_score' => 50,
                'marks_per_question' => 10,
                'target_audience' => 'student',
                'question_type' => 'essay',
                'question_ids' => array_slice($essayQuestions, 0, 5),
            ];
        }
        
        $exams = [];
        foreach ($examsData as $data) {
            $exams[] = Exam::firstOrCreate(
                ['code' => $data['code']],
                [
                    'title' => $data['title'],
                    'description' => $data['description'],
                    'instructions' => $data['instructions'],
                    'duration_minutes' => $data['duration_minutes'],
                    'total_questions' => $data['total_questions'],
                    'passing_score' => $data['passing_score'],
                    'marks_per_question' => $data['marks_per_question'] ?? 1,
                    'negative_marking' => 0,
                    'question_selection_type' => 'manual',
                    'question_type' => $data['question_type'] ?? 'mcq',
                    'question_ids' => $data['question_ids'],
                    'target_audience' => $data['target_audience'],
                    'target_period' => '2024/2025',
                    'target_department' => 'Nursing',
                    'shuffle_questions' => ($data['question_type'] ?? 'mcq') === 'mcq',
                    'shuffle_options' => false,
                    'show_result_immediately' => true,
                    'allow_review' => true,
                    'show_correct_answers' => false,
                    'is_active' => true,
                    'created_by' => $admin->id,
                ]
            );
        }
        
        return $exams;
    }
    
    private function createExamSessions(array $exams, User $admin): array
    {
        $sessions = [];
        
        foreach ($exams as $exam) {
            // Create an active session (available now)
            $sessions[] = ExamSession::firstOrCreate(
                ['exam_id' => $exam->id, 'title' => 'Morning Session'],
                [
                    'start_time' => now()->subHours(1),
                    'end_time' => now()->addHours(5),
                    'max_attempts' => 1,
                    'status' => ExamSessionStatus::ACTIVE,
                    'access_code' => strtoupper(Str::random(6)),
                    'created_by' => $admin->id,
                ]
            );
            
            // Create an upcoming session
            $sessions[] = ExamSession::firstOrCreate(
                ['exam_id' => $exam->id, 'title' => 'Afternoon Session'],
                [
                    'start_time' => now()->addDays(1)->setHour(14),
                    'end_time' => now()->addDays(1)->setHour(18),
                    'max_attempts' => 1,
                    'status' => ExamSessionStatus::SCHEDULED,
                    'access_code' => strtoupper(Str::random(6)),
                    'created_by' => $admin->id,
                ]
            );
        }
        
        return $sessions;
    }
    
    private function createExamAttempts(array $sessions, array $students, array $questions): array
    {
        $attempts = [];
        
        // Only create attempts for the first session (active)
        $session = $sessions[0] ?? null;
        if (!$session) return $attempts;
        
        $exam = $session->exam;
        $examQuestionIds = $exam->question_ids ?? [];
        
        // Create completed attempts for first 3 students
        for ($i = 0; $i < min(3, count($students)); $i++) {
            $student = $students[$i];
            
            // Check if attempt already exists
            $existingAttempt = ExamAttempt::where('exam_session_id', $session->id)
                ->where('candidate_type', Student::class)
                ->where('candidate_id', $student->id)
                ->first();
            
            if ($existingAttempt) {
                $attempts[] = $existingAttempt;
                continue;
            }
            
            $totalQuestions = count($examQuestionIds);
            $correctAnswers = rand(intval($totalQuestions * 0.5), $totalQuestions);
            $wrongAnswers = $totalQuestions - $correctAnswers;
            $scorePercentage = ($correctAnswers / $totalQuestions) * 100;
            
            $attempt = ExamAttempt::create([
                'exam_session_id' => $session->id,
                'candidate_type' => Student::class,
                'candidate_id' => $student->id,
                'question_order' => $examQuestionIds,
                'started_at' => now()->subMinutes(rand(30, 50)),
                'completed_at' => now()->subMinutes(rand(5, 20)),
                'time_spent_seconds' => rand(1200, 2400),
                'total_questions' => $totalQuestions,
                'answered_questions' => $totalQuestions,
                'correct_answers' => $correctAnswers,
                'wrong_answers' => $wrongAnswers,
                'skipped_questions' => 0,
                'score_obtained' => $correctAnswers,
                'score_percentage' => $scorePercentage,
                'passed' => $scorePercentage >= $exam->passing_score,
                'status' => ExamAttemptStatus::COMPLETED,
                'current_question_index' => $totalQuestions,
                'ip_address' => '192.168.1.' . rand(1, 254),
            ]);
            
            // Create answers for this attempt
            foreach ($examQuestionIds as $index => $questionId) {
                $question = collect($questions)->firstWhere('id', $questionId);
                if (!$question) continue;
                
                $isCorrect = $index < $correctAnswers;
                $options = $question->options ?? [];
                $correctAnswer = $question->correct_answer;
                
                // If correct, use correct answer; otherwise, pick a random wrong answer
                if ($isCorrect) {
                    $selectedAnswer = $correctAnswer;
                } else {
                    $optionKeys = collect($options)->pluck('key')->filter(fn($k) => $k !== $correctAnswer)->values();
                    $selectedAnswer = $optionKeys->random() ?? 'A';
                }
                
                ExamAnswer::create([
                    'exam_attempt_id' => $attempt->id,
                    'question_bank_id' => $questionId,
                    'question_index' => $index + 1,
                    'selected_answer' => $selectedAnswer,
                    'correct_answer' => $correctAnswer,
                    'is_correct' => $isCorrect,
                    'marks_obtained' => $isCorrect ? 1 : 0,
                    'marks_possible' => 1,
                    'time_spent_seconds' => rand(30, 180),
                    'answered_at' => now()->subMinutes(rand(5, 30)),
                    'is_skipped' => false,
                ]);
            }
            
            $attempts[] = $attempt;
        }
        
        return $attempts;
    }
}
