<?php

use App\Livewire\Setup\SetupWizard;
use App\Livewire\Exam\ExamList;
use App\Livewire\Exam\TakeExam;
use App\Livewire\Exam\ExamResult;
use App\Livewire\Dashboard\StudentDashboard;
use App\Livewire\Dashboard\StaffDashboard;
use App\Livewire\Dashboard\ApplicantDashboard;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Auth\StudentAuthController;
use App\Http\Controllers\Auth\StaffAuthController;
use App\Http\Controllers\Auth\ApplicantAuthController;

/*
|--------------------------------------------------------------------------
| CSRF Token Refresh Route
|--------------------------------------------------------------------------
| This route is used to refresh the CSRF token and keep the session alive.
| It's called periodically by JavaScript to prevent "Page Expired" errors.
*/
Route::get('/csrf-refresh', function () {
    return response()->json([
        'token' => csrf_token(),
    ]);
})->name('csrf.refresh');

/*
|--------------------------------------------------------------------------
| Setup Routes
|--------------------------------------------------------------------------
| Routes for the initial system setup wizard. These routes are only
| accessible before the setup is complete.
*/
Route::middleware(['setup.incomplete'])->group(function () {
    Route::get('/setup', SetupWizard::class)->name('setup.index');
});

/*
|--------------------------------------------------------------------------
| Main Application Routes
|--------------------------------------------------------------------------
| All routes below this point require setup to be complete.
*/
Route::middleware(['setup.complete'])->group(function () {
    // Root route - redirect to login or dashboard
    Route::get('/', function () {
        // Check if user is already authenticated
        if (auth('student')->check()) {
            return redirect()->route('student.dashboard');
        }
        if (auth('staff')->check()) {
            return redirect()->route('staff.dashboard');
        }
        if (auth('applicant')->check()) {
            return redirect()->route('applicant.dashboard');
        }
        return view('auth.unified-login');
    })->name('login');

    // Authentication Routes
    Route::middleware(['guest.cbt', 'no.cache'])->group(function () {
        // Unified login page (single form for student, staff, and applicant)
        Route::get('/login', function () {
            return view('auth.unified-login');
        })->name('auth.login');
        
        // Student authentication
        Route::post('/student/login', [StudentAuthController::class, 'login'])->name('student.login');
        
        // Staff authentication  
        Route::post('/staff/login', [StaffAuthController::class, 'login'])->name('staff.login');
        
        // Applicant authentication
        Route::post('/applicant/login', [ApplicantAuthController::class, 'login'])->name('applicant.login');
    });

    // Logout routes
    Route::post('/student/logout', [StudentAuthController::class, 'logout'])->name('student.logout');
    Route::post('/staff/logout', [StaffAuthController::class, 'logout'])->name('staff.logout');
    Route::post('/applicant/logout', [ApplicantAuthController::class, 'logout'])->name('applicant.logout');

    // Protected Student Routes
    Route::middleware(['auth:student', 'no.cache'])->group(function () {
        Route::get('/student/dashboard', StudentDashboard::class)->name('student.dashboard');

        // Student Exam Routes
        Route::get('/exams', ExamList::class)->name('exam.list');
        Route::get('/exams/{session}/take', TakeExam::class)->name('exam.take');
        Route::get('/exams/result/{attempt}', ExamResult::class)->name('exam.result');
    });

    // Protected Staff Routes  
    Route::middleware(['auth:staff', 'no.cache'])->group(function () {
        Route::get('/staff/dashboard', StaffDashboard::class)->name('staff.dashboard');

        // Staff Exam Routes (staff can also take exams)
        Route::get('/staff/exams', ExamList::class)->name('staff.exam.list');
        Route::get('/staff/exams/{session}/take', TakeExam::class)->name('staff.exam.take');
        Route::get('/staff/exams/result/{attempt}', ExamResult::class)->name('staff.exam.result');

        // Staff Management Routes (for staff with higher roles)
        Route::get('/staff/grading', \App\Livewire\Staff\EssayGrading::class)->name('staff.grading');
    });

    // Protected Applicant Routes  
    Route::middleware(['auth:applicant', 'no.cache'])->group(function () {
        Route::get('/applicant/dashboard', ApplicantDashboard::class)->name('applicant.dashboard');

        // Applicant Exam Routes (applicants can take recruitment exams)
        Route::get('/applicant/exams', ExamList::class)->name('applicant.exam.list');
        Route::get('/applicant/exams/{session}/take', TakeExam::class)->name('applicant.exam.take');
        Route::get('/applicant/exams/result/{attempt}', ExamResult::class)->name('applicant.exam.result');
    });

    // Staff template download route
    Route::get('/staff/template', function () {
        $headers = [
            'staff_number',
            'full_name',
            'email',
            'phone',
            'department',
            'position',
            'staff_type',
            'employment_status',
            'password'
        ];

        $sampleData = [
            [
                'PN001',
                'Dr. John Doe',
                'john.doe@fnph.gov.ng',
                '+234-801-234-5678',
                'Psychiatry',
                'Consultant Psychiatrist',
                'senior',
                'active',
                'password123'
            ],
            [
                'JN001',
                'Jane Smith',
                'jane.smith@fnph.gov.ng',
                '+234-802-345-6789',
                'Nursing',
                'Registered Nurse',
                'junior',
                'active',
                'securepass456'
            ]
        ];

        $filename = 'staff_import_template.csv';
        $handle = fopen('php://output', 'w');

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        // Write headers
        fputcsv($handle, $headers);

        // Write sample data
        foreach ($sampleData as $row) {
            fputcsv($handle, $row);
        }

        fclose($handle);
        exit;
    })->name('staff.template');

    // Student template download route
    Route::get('/students/template', function () {
        $headers = [
            'full_name',
            'matriculation_number',
            'session',
            'email',
            'password'
        ];

        $sampleData = [
            [
                'John Doe',
                'STU/2024/001',
                '2024/25',
                'john.doe@student.fnph.edu.ng',
                'password123'
            ],
            [
                'Jane Smith',
                'STU/2024/002',
                '2024/25',
                'jane.smith@student.fnph.edu.ng',
                'password123'
            ],
            [
                'Mike Johnson',
                'STU/2025/003',
                '2025/26',
                'mike.johnson@student.fnph.edu.ng',
                'password123'
            ]
        ];

        $filename = 'student_import_template.csv';
        $handle = fopen('php://output', 'w');

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        // Write headers
        fputcsv($handle, $headers);

        // Write sample data
        foreach ($sampleData as $row) {
            fputcsv($handle, $row);
        }

        fclose($handle);
        exit;
    })->name('students.template');

    // Applicant template download route
    Route::get('/applicant/template', function () {
        $headers = [
            'application_number',
            'full_name',
            'position_applied',
            'department',
            'email',
            'phone',
            'password'
        ];

        $sampleData = [
            [
                'APP-2026-001',
                'John Doe',
                'Medical Officer',
                'Psychiatry',
                'john.doe@email.com',
                '+234 801 234 5678',
                'password123'
            ],
            [
                'APP-2026-002',
                'Jane Smith',
                'Staff Nurse',
                'Nursing Services',
                'jane.smith@email.com',
                '+234 802 345 6789',
                'password123'
            ],
            [
                'APP-2026-003',
                'Mike Johnson',
                'IT Support Officer',
                'ICT',
                'mike.johnson@email.com',
                '+234 803 456 7890',
                'password123'
            ]
        ];

        $filename = 'applicant_import_template.csv';
        $handle = fopen('php://output', 'w');

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        // Write headers
        fputcsv($handle, $headers);

        // Write sample data
        foreach ($sampleData as $row) {
            fputcsv($handle, $row);
        }

        fclose($handle);
        exit;
    })->name('applicants.template');
});